module DependencyGenerator

using ..Ahorn, Maple

function getModRoot(fn::String)
    modFolders = Ahorn.getCelesteModDirs()
    modZips = Ahorn.getCelesteModZips()

    allMods = [modFolders; modZips]

    for path in allMods
        checkpath = path
        if !endswith(path, ".zip")
            checkpath = path * (Sys.iswindows() ? "\\" : "/")
        end
        if startswith(lowercase(fn), lowercase(checkpath))
            return path
        end
    end
end

function buildTypeDict(paths::Array{String}, targetType::Type)
    res = Dict{Type, String}()
    for path in paths
        splitPath = split(path, ":")
        rootPath = join(splitPath[1:length(splitPath)-1], ":")

        if haskey(Ahorn.loadedModules, path)
            modul = Ahorn.loadedModules[path]
            for fieldName in names(modul, all = true)
                local entityType = getfield(modul, Symbol(fieldName))
                if isa(entityType, Type) && entityType <: targetType
                    res[entityType] = path
                end
            end
        end
    end
    return res
end

function getModData(path::String)
    fileStr = ""
    if !Ahorn.hasExt(path, ".zip")
        target = joinpath(path, "everest.yaml")
        target2 = joinpath(path, "everest.yml")
        if isfile(target)
            f = open(target, "r")
            fileStr = read(f, String)
            close(f)
        elseif isfile(target2)
            f = open(target2, "r")
            fileStr = read(f, String)
            close(f)
        else
            return (false, nothing)
        end
    else
        zipfh = Ahorn.ZipFile.Reader(path)
        for file in zipfh.files
            if file.name == "everest.yaml" || file.name == "everest.yml"
                fileStr = read(file, String)
                break
            end
        end
        close(zipfh)
    end
    if fileStr != ""
        cleanStr = strip(replace(fileStr, "\uFEFF" => ""))
        data = Ahorn.YAML.load(cleanStr)
        return (true, data)
    else
        return (false, nothing)
    end
end

function generateDependencies()
    println("[Dependency Generator] == Loading current mod ==")

    currentRoot = getModRoot(Ahorn.loadedState.filename)
    if currentRoot == nothing || !isdir(joinpath(currentRoot, "Maps"))
        println("[Dependency Generator] !ERROR! No folder found for the current mod")
        Ahorn.warn_dialog("No folder found for the current mod - mod must be inside Celeste's Mods directory!")
        return
    end

    println("[Dependency Generator] == Building tables ==")

    local foundMods = String[]
    println("[Dependency Generator] Loading entities")
    local entityToPath = buildTypeDict(Ahorn.loadedEntities, Maple.Entity)
    println("[Dependency Generator] Loading triggers")
    local triggerToPath = buildTypeDict(Ahorn.loadedTriggers, Maple.Trigger)
    println("[Dependency Generator] Loading effects")
    local effectToPath = buildTypeDict(Ahorn.loadedEffects, Maple.Effect)

    println("[Dependency Generator] == Finding used dependencies ==")

    for (root, dirs, files) in walkdir(joinpath(currentRoot, "Maps")), file in files
        if !Ahorn.hasExt(file, ".bin")
            continue
        end
        println("[Dependency Generator] Searching map: $file")
        fullPath = joinpath(root, file)
        currentMap = nothing
        if fullPath == Ahorn.loadedState.filename
            currentMap = Ahorn.loadedState.map
        else
            currentMap = loadMap(fullPath)
        end
        for room in currentMap.rooms, entity in room.entities
            if haskey(entityToPath, typeof(entity))
                root = getModRoot(entityToPath[typeof(entity)])
                if root != nothing && !(root in foundMods) && root != currentRoot
                    push!(foundMods, root)
                end
            end
        end
        for room in currentMap.rooms, trigger in room.triggers
            if haskey(triggerToPath, typeof(trigger))
                root = getModRoot(triggerToPath[typeof(trigger)])
                if root != nothing && !(root in foundMods) && root != currentRoot
                    push!(foundMods, root)
                end
            end
        end
        for room in currentMap.rooms, decal in [room.fgDecals; room.bgDecals]
            decalTexture = endswith(decal.texture, ".png") ? decal.texture[1:length(decal.texture)-4] : decal.texture
            spritePath = Ahorn.findExternalSprite("decals/$decalTexture")
            if spritePath != nothing
                root = getModRoot(spritePath)
                if root != nothing && !(root in foundMods) && root != currentRoot
                    push!(foundMods, root)
                end
            end
        end
        for effect in [currentMap.style.foregrounds; currentMap.style.backgrounds]
            if haskey(effectToPath, typeof(effect))
                root = getModRoot(effectToPath[typeof(effect)])
                if root != nothing && !(root in foundMods) && root != currentRoot
                    push!(foundMods, root)
                end
            elseif isa(effect, Maple.Parallax)
                effectTexture = endswith(effect.texture, ".png") ? effect.texture[1:length(effect.texture)-4] : effect.texture
                spritePath = Ahorn.findExternalSprite(effect.texture)
                if spritePath != nothing
                    root = getModRoot(spritePath)
                    if root != nothing && !(root in foundMods) && root != currentRoot
                        push!(foundMods, root)
                    end
                end
            end
        end
    end

    println("[Dependency Generator] == Parsing YAMLs ==")

    parsedDependencies = Dict{Any,Any}[Dict{Any,Any}(
            "Name" => "Everest",
            "Version" => "1.0.0"
        )]
    for modPath in foundMods
        succ, data = getModData(modPath)
        if succ
            dependency = Dict{Any,Any}(
                    "Name" => get(data[1], "Name", "???"),
                    "Version" => get(data[1], "Version", "1.0.0")
                )
            push!(parsedDependencies, dependency)
            println("[Dependency Generator] Parsed dependency: " * dependency["Name"] * " (" * dependency["Version"] * ")")
        end
    end

    succ, data = getModData(currentRoot)
    if !succ
        println("[Dependency Generator] No YAML found, creating one")
        data = Dict{Any,Any}[Dict{Any,Any}(
                "Name" => basename(currentRoot),
                "Version" => "1.0.0"
            )]
    end
    if !haskey(data[1], "Dependencies")
        data[1]["Dependencies"] = Dict{Any,Any}[]
    end
    dependencyData = data[1]["Dependencies"]

    println("[Dependency Generator] == Updating dependencies ==")

    existingDeps = Dict{Any,Any}()
    for dependency in dependencyData
        existingDeps[dependency["Name"]] = dependency
    end
    for dependency in parsedDependencies
        if haskey(existingDeps, dependency["Name"])
            existing = existingDeps[dependency["Name"]]
            oldver = VersionNumber(existing["Version"])
            newver = VersionNumber(dependency["Version"])
            if newver > oldver && Ahorn.ask_dialog("Found newer version of $(dependency["Name"]). Update?\n$oldver => $newver", Ahorn.window)
                existing["Version"] = dependency["Version"]
            end
            delete!(existingDeps, dependency["Name"])
        else
            push!(dependencyData, dependency)
        end
    end
    for (name, dependency) in existingDeps
        if Ahorn.ask_dialog("Existing dependency $name appears unused. Remove?", Ahorn.window)
            filter!(x -> x["Name"] != name, dependencyData)
        end
    end

    println("[Dependency Generator] == Writing YAML ==")

    if succ
        println("[Dependency Generator] Creating backup")
        cp(joinpath(currentRoot, "everest.yaml"), joinpath(currentRoot, "everest.yaml-backup"), force = true)
    end

    f = open(joinpath(currentRoot, "everest.yaml"), "w")
    write(f, write_yaml(data, "", ["Name", "Version", "DLL", "Dependencies", "OptionalDependencies"], ["Name", "Version"]))
    close(f)

    println("[Dependency Generator] == Done! ==")
    Ahorn.warn_dialog("everest.yaml generated!", Ahorn.window)
end

function generateDependenciesOption()
    Ahorn.eval(Meta.parse("DependencyGenerator.generateDependencies()"))
end

for choice in Ahorn.menubarChoices
    if choice.name == "Map"
        optname = "Generate everest.yaml (Plugin)"
        if length(findall(x -> isa(x, Ahorn.Menubar.MenuChoice) && x.name == optname, choice.children)) == 0
            push!(choice.children, Ahorn.Menubar.MenuChoice(optname, (w) -> generateDependenciesOption()))
            break
        end
    end
end


#####################
#
#  Modified YAML code
#
#####################


function write_yaml(io::IO, data::Any, prefix::AbstractString="", order::Array{String}=String[], no_quote::Array{String}=String[])
    print(io, prefix) # print the prefix, e.g. a comment at the beginning of the file
    _print(io, order, no_quote, data) # recursively print the data
end

function write_yaml(data::Any, prefix::AbstractString="", order::Array{String}=String[], no_quote::Array{String}=String[])
    io = IOBuffer()
    write_yaml(io, data, prefix, order, no_quote)
    return String(take!(io))
end

# recursively print a dictionary
_print(io::IO, order::Array{String}, no_quote::Array{String}, dict::AbstractDict, level::Int=0, ignore_level::Bool=false, use_quote::Bool=true) =
    if length(dict) > 0
        already_printed = Any[]
        for (i, field) in enumerate(order)
            if !(field in already_printed) && haskey(dict, field)
                _print(io, order, no_quote, Pair(field, dict[field]), level, ignore_level ? i == 1 : false, !(field in no_quote))
                push!(already_printed, field)
            end
        end
        for (i, pair) in enumerate(dict)
            if !(pair[1] in already_printed)
                _print(io, order, no_quote, pair, level, ignore_level ? i == 1 : false, !(pair[1] in no_quote)) # ignore indentation of first pair
            end
        end
    else
        @warn "Writing an empty $(typeof(dict)), which might be parsed as nothing"
        print(io, "\n") # https://github.com/JuliaData/YAML.jl/issues/81
    end

# recursively print an array
_print(io::IO, order::Array{String}, no_quote::Array{String}, arr::AbstractVector, level::Int=0, ignore_level::Bool=false, use_quote::Bool=true) =
    for (i, elem) in enumerate(arr)
        if typeof(elem) <: AbstractVector # vectors of vectors must be handled differently
            print(io, _indent("-\n", level))
            _print(io, order, no_quote, elem, level + 1, false, use_quote)
        else
            print(io, _indent("- ", level))   # print the sequence element identifier '-'
            _print(io, order, no_quote, elem, level + 1, true, use_quote) # print the value directly after
        end
    end

# print a single key-value pair
function _print(io::IO, order::Array{String}, no_quote::Array{String}, pair::Pair, level::Int=0, ignore_level::Bool=false, use_quote::Bool=true)
    key = if typeof(pair[1]) == Nothing
        "null" # this is what the YAML parser interprets as 'nothing'
    else
        string(pair[1]) # any useful case
    end
    print(io, _indent(key * ":", level, ignore_level)) # print the key
    if (typeof(pair[2]) <: AbstractDict || typeof(pair[2]) <: AbstractVector)
        print(io, "\n") # a line break is needed before a recursive structure
    else
        print(io, " ") # a whitespace character is needed before a single value
    end
    _print(io, order, no_quote, pair[2], level + 1, false, use_quote) # print the value
end

# _print a single string, which may contain multiple lines
_print(io::IO, order::Array{String}, no_quote::Array{String}, str::AbstractString, level::Int=0, ignore_level::Bool=false, use_quote::Bool=true) =
    if occursin("\n", str) # handle multi-line strings
        indentation = repeat("  ", level + 1)
        println(io, "|\n$indentation" * replace(str, "\n"=>"\n"*indentation)) # indent each line
    else
        if use_quote
            println(io, "\"" * replace(str, "\"" => "\\\"") * "\"") # quote all strings
        else
            println(io, str)
        end
    end

# handle NaNs and Infs
_print(io::IO, order::Array{String}, no_quote::Array{String}, val::Float64, level::Int=0, ignore_level::Bool=false, use_quote::Bool=true) =
    if isfinite(val)
        println(io, string(val)) # the usual case
    elseif isnan(val)
        println(io, ".NaN") # this is what the YAML parser interprets as NaN
    elseif val == Inf
        println(io, ".inf")
    elseif val == -Inf
        println(io, "-.inf")
    end

_print(io::IO, order::Array{String}, no_quote::Array{String}, val::Nothing, level::Int=0, ignore_level::Bool=false, use_quote::Bool=true) =
    println(io, "~") # this is what the YAML parser interprets as nothing

# _print any other single value
_print(io::IO, order::Array{String}, no_quote::Array{String}, val::Any, level::Int=0, ignore_level::Bool=false, use_quote::Bool=true) =
    println(io, string(val)) # no indentation is required

# add indentation to a string
_indent(str::AbstractString, level::Int, ignore_level::Bool=false) =
    repeat("  ", ignore_level ? 0 : level) * str

end