-- table extensions

local expect = require("mods").requireFromPlugin("lib.bitsbolts.expect")
local tablex = {}


function tablex.find(tab, value)
  expect(1, tab, "table")
  for i, v in pairs(tab) do
    if v == value then return i end
  end
  return nil
end

function tablex.findf(tab, func)
  expect(1, tab, "table")
  for k, v in pairs(tab) do
    if func(k, v, tab) then return k end
  end
  return nil
end


function tablex.filter(tab, func)
  expect(1, tab, "table")
  expect(2, func, "function")
  local out = {}
  for k, v in pairs(tab) do
    if func(v, k, tab) then table.insert(out, v) end
  end
  return out
end

function tablex.filterk(tab, func)
  expect(1, tab, "table")
  expect(2, func, "function")
  local out = {}
  for k, v in pairs(tab) do
    if func(v, k, tab) then out[k] = v end
  end
  return out
end


function tablex.fold(tab, default, func)
  expect(1, tab, "table")
  expect(3, func, "function")
  local val = default
  for k, v in ipairs(tab) do
    val = func(v, k, val, tab)
  end
  return val
end

function tablex.count(tab, func)
  expect(1, tab, "table")
  expect(2, func, "function")
  local n = 0
  for k, v in pairs(tab) do
    if func(v, k, n, tab) then n = n + 1 end
  end
  return n
end


function tablex.any(tab, func)
  expect(1, tab, "table")
  expect(2, func, "function")
  for k, v in pairs(tab) do
    if func(v, k, tab) then return true end
  end
  return false
end

function tablex.all(tab, func)
  expect(1, tab, "table")
  expect(2, func, "function")
  for k, v in pairs(tab) do
    if not func(v, k, tab) then return false end
  end
  return true
end

function tablex.clone(val)
  if type(val) == "table" then
    local copy = {}
    for k, v in pairs(val) do
      copy[k] = v
    end

    if getmetatable(val) ~= nil then
      setmetatable(copy, tablex.clone(getmetatable(val)))
    end
    return copy
  else
    return val
  end
end

function tablex.deepclone(val)
  if type(val) == "table" then
    local copy = {}
    for k, v in pairs(val) do
      copy[tablex.deepclone(k)] = tablex.deepclone(v)
    end

    if getmetatable(val) ~= nil then
      setmetatable(copy, tablex.deepclone(getmetatable(val)))
    end
    return copy
  else
    return val
  end
end

function tablex.equal(tab1, tab2)
  expect(1, tab1, "table", "nil")
  expect(2, tab2, "table", "nil")
  local keys1 = {}
  for k, v in pairs(tab1) do
    if (
      (type(v) == "table" and type(tab2[k]) == "table" and tablex.equal(v, tab2[k]))
      or (v ~= tab2[v])
    ) then
      return false
    end
    keys1[k] = true
  end

  -- check if the tables have disjoint keys
  for k, _ in pairs(tab2) do
    if keys1[k] == nil then return false end
  end
  return true
end

function tablex.concat(tab1, tab2)
  expect(1, tab1, "table")
  expect(2, tab2, "table")
  local out = tablex.clone(tab1)
  for k, v in pairs(tab2) do
    out[k] = v
  end
  return out
end

function tablex.keys(tab)
  expect(1, tab, "table")
  local keys = {}
  for key, _ in pairs(tab) do
    table.insert(keys, key)
  end
  return keys
end


function tablex.apply(...)
  local out = {}
  for i, tab in ipairs({...}) do
    expect(i, tab, "table")
    for k, v in pairs(tab) do out[k] = v end
  end
  return out
end


local function escape(str)
  return str:gsub("([\"'\\])", "\\%1")
    :gsub("\a", "\\a"):gsub("\b", "\\b"):gsub("\f", "\\f"):gsub("\n", "\\n")
    :gsub("\r", "\\r"):gsub("\t", "\\t"):gsub("\v", "\\v")
end

function tablex.stringify(tab, debug)
  if expect(2, debug, "boolean", "nil") == nil then debug = true end

  if type(tab) == "string" then
    if debug then return ("\"%s\""):format(escape(tab))
    else return ("\"%s\""):format(tab) end
  end
  if type(tab) ~= "table" then return tostring(tab) end

  local out = ""
  local mt = getmetatable(out)
  if mt ~= nil and type(mt.__name) == "string" then
    out = out .. ("%s:"):format(mt.__name)
  end

  local content = "{ "
  local first = true
  for k, v in pairs(tab) do
    if not first then content = content .. ", " end
    first = false

    if debug then
      content = content .. ("%s = %s"):format(
        tablex.stringify(k, debug),
        tablex.stringify(v, debug)
      )
    else
      content = content .. ("%s = %s"):format(
        tostring(k) or tablex.stringify(k, debug),
        tostring(v) or tablex.stringify(v, debug)
      )
    end
  end
  content = content .. " }"
  if first then content = "{}" end

  return out .. content
end


return tablex
