local rectangle = require("structs.rectangle")
local drawableRectangle = require("structs.drawable_rectangle")
local drawableNinePatch = require("structs.drawable_nine_patch")

local tablex = require("mods").requireFromPlugin("lib.bitsbolts.tablex")
local vec2 = require("mods").requireFromPlugin("lib.bitsbolts.vec2")

local ninepatch_cache = {}
local function gen_ninepatch_from_texture(texture, opts, x, y, w, h)
  local cachedKey = tablex.findf(ninepatch_cache, function(key, value)
    return key.texture == texture and key.w == w and key.h == h
      and tablex.equal(key.opts, opts)
  end)

  if cachedKey ~= nil then return ninepatch_cache[cachedKey] end

  local patch = drawableNinePatch.fromTexture(texture, opts, x, y, w, h)
  ninepatch_cache[{ texture = texture, opts = opts, w = w, h = h }] = patch
  return patch
end

local function gen_divider_sprites(base, size, is_dir_x, left_blue)
  local sprite_size
  local sprite_offset
  if is_dir_x then
    sprite_size = vec2(16, size.y)
    sprite_offset = vec2(-12, 0)
  else
    sprite_size = vec2(size.x, 16)
    sprite_offset = vec2(0, -4)
  end

  local dir_x_symbol = {
    [true] = "x",
    [false] = "y",
  }

  local edge1 = drawableNinePatch.fromTexture(
    "bitsbolts/loenn/portal_edge_" .. dir_x_symbol[is_dir_x],
    {
      mode = "border", borderMode = "random",
      color = {0/255, 0/255, 255/255, 255/255}
    },
    base.x + sprite_offset.x, base.y + sprite_offset.y, sprite_size.x, sprite_size.y
  )

  local edge2 = tablex.deepclone(edge1)
  edge2.color = {255/255, 127/255, 0/255, 255/255}
  edge2.drawX = edge2.drawX + sprite_offset.x/12
  edge2.drawY = edge2.drawY + sprite_offset.y/4

  return {edge1, edge2}
end

local function gen_portal_sprites(origin, size, is_dir_x, left_blue)
  local mainAxis
  local otherAxis
  if is_dir_x then
    mainAxis = vec2(1, 0)
    otherAxis = vec2(0, 1)
  else
    mainAxis = vec2(0, 1)
    otherAxis = vec2(1, 0)
  end

  local midpoint1 = origin + (size:idiv(2)) * mainAxis
  local midpoint2 = midpoint1 + size*otherAxis
  local sprites = gen_divider_sprites(midpoint1, size*otherAxis, is_dir_x, left_blue)

  local edge = gen_ninepatch_from_texture(
    "bitsbolts/loenn/base_region",
    { mode = "border", borderMode = "random" },
    origin.x, origin.y, size.x, size.y
  )

  table.insert(sprites, edge)

  return sprites
end

local exports = {
  name = "Bitsbolts/WorldPortalDual",

  placements = {
    {
      name = "entity",
      data = { height = 72, width = 128, mode = "x", flags = "", nodes = {{}} }
    },
  },

  nodeLimits = {1, 1},
  nodeLineRenderType = "line",
  nodeVisibility = "always",
  minimumSize = {4*8, 4*8},

  fieldInformation = {
    mode = {
      options = {
        { "horizontal", "x" },
        { "vertical", "y" },
      },
      editable = false
    }
  },

  depth = -10001,
  sprite = function(room, entity)
    local origin = vec2(entity.x, entity.y)
    local size = vec2(entity.width, entity.height)
    return gen_portal_sprites(origin, size, entity.mode == "x", entity.node ~= nil)
  end,

  nodeSprite = function(room, entity, node)
    local size = vec2(entity.width, entity.height)
    return gen_portal_sprites(vec2(node.x, node.y), size, entity.mode == "x", true)
  end,

  rectangle = function(room, entity)
    return rectangle.create(
      entity.x, entity.y, entity.width, entity.height
    )
  end,

  nodeRectangle = function(room, entity, node)
    return rectangle.create(
      node.x, node.y, entity.width, entity.height
    )
  end,
}

return exports
