-- table extensions

local expect = require("Assets/Cutscenes/RotationCollab/expect")
local tablex = {}


function tablex.find(tab, value)
  expect(1, tab, "table")
  for i, v in pairs(tab) do
    if v == value then return i end
  end
  return nil
end


function tablex.filter(tab, func)
  expect(1, tab, "table")
  expect(2, func, "function")
  local out = {}
  for k, v in pairs(tab) do
    if func(v, k, tab) then table.insert(out, v) end
  end
  return out
end

function tablex.filterk(tab, func)
  expect(1, tab, "table")
  expect(2, func, "function")
  local out = {}
  for k, v in pairs(tab) do
    if func(v, k, tab) then out[k] = v end
  end
  return out
end


function tablex.fold(tab, default, func)
  expect(1, tab, "table")
  expect(3, func, "function")
  local val = default
  for k, v in ipairs(tab) do
    val = func(v, k, val, tab)
  end
  return val
end

function tablex.count(tab, func)
  expect(1, tab, "table")
  expect(2, func, "function")
  local n = 0
  for k, v in pairs(tab) do
    if func(v, k, n, tab) then n = n + 1 end
  end
  return n
end


function tablex.any(tab, func)
  expect(1, tab, "table")
  expect(2, func, "function")
  for k, v in pairs(tab) do
    if func(v, k, tab) then return true end
  end
  return false
end

function tablex.all(tab, func)
  expect(1, tab, "table")
  expect(2, func, "function")
  for k, v in pairs(tab) do
    if not func(v, k, tab) then return false end
  end
  return true
end


function tablex.keys(tab)
  expect(1, tab, "table")
  local keys = {}
  for key, _ in pairs(tab) do
    table.insert(keys, key)
  end
  return keys
end


function tablex.apply(...)
  local out = {}
  for i, tab in ipairs({...}) do
    expect(i, tab, "table")
    for k, v in pairs(tab) do out[k] = v end
  end
  return out
end


local function escape(str)
  return str:gsub("([\"'\\])", "\\%1")
    :gsub("\a", "\\a"):gsub("\b", "\\b"):gsub("\f", "\\f"):gsub("\n", "\\n")
    :gsub("\r", "\\r"):gsub("\t", "\\t"):gsub("\v", "\\v")
end

function tablex.stringify(tab, debug)
  if expect(2, debug, "boolean", "nil") == nil then debug = true end

  if type(tab) == "string" then
    if debug then return ("\"%s\""):format(escape(tab))
    else return ("\"%s\""):format(tab) end
  end
  if type(tab) ~= "table" then return tostring(tab) end

  local out = ""
  local mt = getmetatable(out)
  if mt ~= nil and type(mt.__name) == "string" then
    out = out .. ("%s:"):format(mt.__name)
  end

  local content = "{ "
  local first = true
  for k, v in pairs(tab) do
    if not first then content = content .. ", " end
    first = false

    if debug then
      content = content .. ("%s = %s"):format(
        tablex.stringify(k, debug),
        tablex.stringify(v, debug)
      )
    else
      content = content .. ("%s = %s"):format(
        tostring(k) or tablex.stringify(k, debug),
        tostring(v) or tablex.stringify(v, debug)
      )
    end
  end
  content = content .. " }"
  if first then content = "{}" end

  return out .. content
end


return tablex
