using System;
using System.Linq;
using System.Collections.Generic;
using System.Threading.Tasks;
using Buttplug;
using Celeste.Mod.CelesteButtplug.Settings.Multiplayer;
using YamlDotNet.Serialization;
// ReSharper disable MemberCanBePrivate.Global
// ReSharper disable PropertyCanBeMadeInitOnly.Global

namespace Celeste.Mod.CelesteButtplug.Settings;

public class CelesteButtplugModuleSettings : EverestModuleSettings {
    
    [SettingSubMenu]
    public class ActuatorTypeSubMenu
    {
        public bool Vibrate { get; set; } = true;
        public bool Rotate { get; set; }
        public bool Oscillate { get; set; }
        public bool Constrict { get; set; }
        public bool Inflate { get; set; }
        
        public IEnumerable<ActuatorType> GetActuatorTypes()
        {
            var ret = new HashSet<ActuatorType>();
            if(Vibrate) ret.Add(ActuatorType.Vibrate);
            if(Oscillate) ret.Add(ActuatorType.Oscillate);
            if(Constrict) ret.Add(ActuatorType.Constrict);
            if(Inflate) ret.Add(ActuatorType.Inflate);
            return ret;
        }
    }

    [YamlIgnore]
    public bool Connect { get; set; }
    public void CreateConnectEntry(TextMenu menu, bool inGame)
    {
        TextMenu.OnOff connectEntry = new(
            label: "Connect",
            on: CelesteButtplugModule.Manager.Connected()
        );
        menu.Add(connectEntry);

        connectEntry.Change(newValue =>
        {
            connectEntry.Label = "Connect - ...";
            connectEntry.Disabled = true;
            Task.Run(async Task () =>
            {
                try
                {
                    if (newValue)
                    {
                        await CelesteButtplugModule.Manager.Connect(GetUri());
                        connectEntry.Label = "Connect - Successful";
                    }
                    else
                    {
                        await CelesteButtplugModule.Manager.Disconnect();
                        connectEntry.Label = "Connect - Disconnected";
                    }
                }
                catch (Exception e)
                {
                    Logger.LogDetailed(e);
                    connectEntry.Label = "Connect - Exception occured, check logs...";
                
                    connectEntry.PreviousIndex = connectEntry.Index;
                    connectEntry.Index = 1 - connectEntry.Index;
                    connectEntry.lastDir = connectEntry.Index == 1 ? 1 : -1;
                    connectEntry.ValueWiggler.Start();
                }
                connectEntry.Disabled = false;
            });
        });
    }
    public bool AutomaticallyConnect { get; set; }
    [SettingMinLength(7)]
    [SettingMaxLength(21)]
    public string IntifaceConnection { get; set; } = "127.0.0.1:12345";
    public Uri GetUri()
    {
        return new Uri("ws://" + IntifaceConnection + "/");
    }

    public ActuatorTypeSubMenu ActuatorTypes { get; set; } = new();
    public MultiplayerDeathTriggerSubMenu OnDie { get; set; } = new()
    {
        Enabled = true,
    };
    public MultiplayerDashTriggerSubMenu OnDash { get; set; } = new();
    public MultiplayerJumpTriggerSubMenu OnJump { get; set; } = new();
    [SettingSubText("Recommended to keep settings low on this one")]
    public MultiplayerSqueeshTriggerSubMenu OnSqueesh { get; set; } = new();
    public MultiplayerPickupTriggerSubMenu OnPickup { get; set; } = new();
    public MultiplayerThrowTriggerSubMenu OnThrow { get; set; } = new();
    public TriggerSubMenu OnScreenTransition { get; set; } = new();
    public TriggerSubMenu OnBerryCollected { get; set; } = new();
    [SettingSubText("THIS ONE TRIGGERS A LOT: Recommended to keep settings REALLY low on this one")]
    public TriggerSubMenu WhileClimbing { get; set; } = new()
    {
        OverrideGlobalMillis = true,
        Millis = 1,
    };
    
    [SettingRange(min: 1, max: 2000, largeRange: true)]
    public int Millis { get; set; } = 1000;
    [SettingRange(min: 0, max: 100, largeRange: true)]
    public int Strength { get; set; } = 100;
    
    [YamlIgnore]
    public bool TryOut { get; set; }
    public void CreateTryOutEntry(TextMenu menu, bool inGame)
    {
        TextMenu.OnOff tryOutEntry = new(
            label: "Try out",
            on: false
        );
        menu.Add(tryOutEntry);

        tryOutEntry.Change(_ =>
        {
            if (!CelesteButtplugModule.Manager.Connected())
            {
                tryOutEntry.Label = "Try out - Not connected";
                tryOutEntry.PreviousIndex = tryOutEntry.Index;
                tryOutEntry.Index = 1 - tryOutEntry.Index;
                tryOutEntry.lastDir = tryOutEntry.Index == 1 ? 1 : -1;
                tryOutEntry.ValueWiggler.Start();
                return;
            }
            tryOutEntry.Disabled = true;
            Task.Run(async Task () =>
            {
                tryOutEntry.Label = "Try out - Starts in 1s! (" + TryOutSettings.GetFinalStrength() + "% strength, " + TimeSpan.FromMilliseconds(TryOutSettings.GetFinalMillis()).TotalSeconds + " seconds)";
                await Task.Delay(1000);
                await Task.WhenAll(CelesteButtplugModule.Manager.Devices().Select(async device => await CelesteButtplugModule.Manager.TriggerBuzzAsync(device, TryOutSettings)));
                tryOutEntry.PreviousIndex = tryOutEntry.Index;
                tryOutEntry.Index = 1 - tryOutEntry.Index;
                tryOutEntry.lastDir = tryOutEntry.Index == 1 ? 1 : -1;
                tryOutEntry.Disabled = false;
                tryOutEntry.ValueWiggler.Start();
                tryOutEntry.Label = "Try out - Done";
            });
        });
    }
    [YamlIgnore]
    public TriggerSubMenu TryOutSettings { get; set; } = new()
    {
        OverrideGlobalMillis = true,
        OverrideGlobalStrength = true,
        Millis = 2000,
        Strength = 25,
    };
}