﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Numerics;
using System.Reflection;
using Monocle;
using MonoMod.Cil;
using MonoMod.RuntimeDetour;

namespace Celeste.Mod.CelesteButtplug.Managers;

public class MultiplayerHooksManager
{
    private const string ModName = "CelesteNet.Client";
    private const string ModVersions = "2.4.2";

    private static Type _ghostType;

    public bool IsCompatible { get; private set; }
    private List<ILHook> _hooks = [];

    public MultiplayerHooksManager()
    {
        CheckCompatibility();
    }

    private bool CheckCompatibility()
    {
        if (IsCompatible) return true;
        
        IEnumerable<EverestModule> celesteNetModules = Everest.Modules.Where(module => module.Metadata.Name == ModName).ToList();
        if (!celesteNetModules.Any())
        {
            Logger.Debug(nameof(CelesteButtplugModule), "No " + ModName + " found");
            return false;
        }
        string[] compatibleModVersions = ModVersions.Split(',');
        IEnumerable<EverestModule> compatibleCelesteNetModules = celesteNetModules.Where(module => compatibleModVersions.Any(s => module.Metadata.VersionString == s));
        if (!compatibleCelesteNetModules.Any())
        {
            Logger.Debug(nameof(CelesteButtplugModule), ModName + " found, but incompatible... (found: " + string.Join(", ", celesteNetModules.Select(m => m.Metadata.VersionString)) + ")");
            return false;
        }
        
        IsCompatible = true;
        return IsCompatible;
    }

    public void Register()
    {
        if (!CheckCompatibility()) return;
        Logger.Debug(nameof(CelesteButtplugModule), "Hooking into methods");
        _ghostType = Type.GetType("Celeste.Mod.CelesteNet.Client.Entities.Ghost, " + ModName) ?? throw new NullReferenceException();
        
        AddHook(_ghostType.GetMethod("HandleDeath", BindingFlags.Public | BindingFlags.Instance), il =>
        {
            ILCursor cursor = new(il);
            if (cursor.TryGotoNext(MoveType.After, instruction => instruction.MatchCallOrCallvirt(typeof(Scene).GetMethod("Add", [typeof(Entity)]))))
            {
                // cursor.EmitLdarg0(); // "this"
                cursor.EmitCall(typeof(MultiplayerEvents).GetMethod("Death", BindingFlags.NonPublic | BindingFlags.Static));
                return;
            }
            Logger.Error(nameof(CelesteButtplugModule), "Couldn't apply IL modifications to HandleDeath");
        });
        On.Celeste.Level.Update += LevelOnUpdate;
    }

    private void AddHook(MethodInfo method, ILContext.Manipulator manipulator)
    {
        ArgumentNullException.ThrowIfNull(method);
        ArgumentNullException.ThrowIfNull(manipulator);

        Logger.Debug(nameof(CelesteButtplugModule), "Hooking into " + method.Name);
        _hooks.Add(new ILHook(method, manipulator));
    }
    
    public void Unregister()
    {
        foreach (ILHook hook in _hooks)
        {
            hook.Dispose();
        }
        On.Celeste.Level.Update -= LevelOnUpdate;
    }

    private static Dictionary<Entity, string> _prevAnimations = new();
    private static void LevelOnUpdate(On.Celeste.Level.orig_Update orig, Level self)
    {
        foreach (Entity entity in self.Entities.Where(entity => self.Bounds.Contains((int)entity.X, (int)entity.Y)))
        {
            Type entityType = entity.GetType();
            if (entityType != _ghostType) continue;
            Sprite sprite = (Sprite)entityType.GetField("Sprite", BindingFlags.Public | BindingFlags.Instance)?.GetValue(entity);
            if (sprite == null) continue;
            if (_prevAnimations.TryGetValue(entity, out string value) && value == sprite.LastAnimationID) continue;
            _prevAnimations[entity] = sprite.LastAnimationID;
            
            object playerInfo = entityType.GetField("PlayerInfo")?.GetValue(entity);
            string playerName = (string) playerInfo?.GetType().GetField("FullName")?.GetValue(playerInfo);
            
            Logger.Debug(nameof(CelesteButtplugModule), playerName + ": " + sprite.LastAnimationID);
            switch (sprite.LastAnimationID)
            {
                case "dash":
                    MultiplayerEvents.Dash();
                    break;
                case "jumpSlow":
                case "jumpFast":
                    MultiplayerEvents.Jump();
                    break;
                case "duck":
                    MultiplayerEvents.Squeesh();
                    break;
                case "pickUp":
                    MultiplayerEvents.Pickup();
                    break;
                case "throw":
                    MultiplayerEvents.Throw();
                    break;
            }
        }
        orig(self);
    }
}