﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Buttplug;
using Buttplug.NewtonsoftJson;
using Celeste.Mod.CelesteButtplug.Settings;

namespace Celeste.Mod.CelesteButtplug.Managers;

public class ButtplugClientManager(CelesteButtplugModuleSettings settings)
{
    private ButtplugClient Client { get; } = new("CelesteButtplug", new ButtplugNewtonsoftJsonConverter());
    private CancellationTokenSource _tokenSource;

    public async Task Connect(Uri uri) {
        try
        {
            await Client.ConnectAsync(uri, CancellationToken.None);
            Logger.Info(nameof(CelesteButtplugModule), "Successfully connected to " + uri);
        }
        catch (Exception)
        {
            Logger.Error(nameof(CelesteButtplugModule), "Unable to connect, perhaps Initface Central hasn't started yet?");
            throw;
        }
    }

    public async Task Disconnect() {
        if(!Connected()) return;
        await Client.StopAllDevicesAsync(CancellationToken.None);
        Logger.Debug(nameof(CelesteButtplugModule), "All devices: off");
        await Client.DisconnectAsync();
        Logger.Info(nameof(CelesteButtplugModule), "Successfully disconnected");
    }

    public ICollection<ButtplugDevice> Devices()
    {
        return Connected() ? Client.Devices : new List<ButtplugDevice>();
    }

    public bool Connected()
    {
        return Client.IsConnected;
    }

    public async Task<bool> TryReconnect()
    {
        if (Connected()) return true;
        if (!settings.AutomaticallyConnect) return false;
        try
        {
            await Connect(settings.GetUri());
            return true;
        }
        catch (Exception)
        {
            // ignored
            return false;
        }
    }
    
    public async Task TriggerBuzzAsync(ButtplugDevice device, TriggerSubMenu triggerSubMenu)
    {
        if(!await TryReconnect()) return;
        if (_tokenSource != null)
        {
            Logger.Verbose(nameof(CelesteButtplugModule), "Cancel anything still running");
            await _tokenSource.CancelAsync();
        }
        _tokenSource = new CancellationTokenSource();
        
        int millis = triggerSubMenu.GetFinalMillis();
        double strength = (double)triggerSubMenu.GetFinalStrength() / 100;
        DateTime stopAt = DateTime.Now.AddMilliseconds(millis);

        await Task.Run(() => TaskFunctions.WaitWhile(() => stopAt > DateTime.Now, async () =>
        {
            double scalar = GetScalar(strength, triggerSubMenu.EasingCurve, stopAt, millis);
            await TriggerBuzzAsync(device, scalar);
        }, 2, -1, _tokenSource.Token));

        await TriggerBuzzAsync(device, 0); // Turn off
    }
    
    private async Task TriggerBuzzAsync(ButtplugDevice device, double scalar) 
    {
        if(!Connected()) return;
        
        List<ActuatorType> types = settings.ActuatorTypes.GetActuatorTypes().ToList();
        
        string stringifiedTypes = string.Join(", ", types);
        if(settings.ActuatorTypes.Rotate) stringifiedTypes += (stringifiedTypes != "" ? ", " : "") + "Rotate";
        
        if(stringifiedTypes != "") Logger.Verbose(nameof(CelesteButtplugModule), stringifiedTypes + ": " + scalar);
        if (settings.ActuatorTypes.Rotate)
        {
            foreach (ButtplugDeviceRotateActuator actuator in device.RotateActuators)
            {
                try
                {
                    await actuator.RotateAsync(scalar, false, CancellationToken.None).ConfigureAwait(true);
                }
                catch (Exception e)
                {
                    Logger.Error(nameof(CelesteButtplugModule), "Rotate failed:");
                    if (e is TaskCanceledException or OperationCanceledException)
                    {
                        Logger.Debug(nameof(CelesteButtplugModule), "Task cancelled");
                        return;
                    }
                    Logger.LogDetailed(e);
                }
            }
        }
        
        foreach (ButtplugDeviceScalarActuator actuator in device.ScalarActuators.Where(actuator => types.Contains(actuator.ActuatorType)))
        {
            try
            {
                await actuator.ScalarAsync(scalar, CancellationToken.None).ConfigureAwait(true);
            }
            catch (Exception e)
            {
                Logger.Error(nameof(CelesteButtplugModule), actuator.ActuatorType + " failed:");
                if (e is TaskCanceledException or OperationCanceledException)
                {
                    Logger.Debug(nameof(CelesteButtplugModule), "Task cancelled");
                    return;
                }
                Logger.LogDetailed(e);
            }
        }
    }
    
    private static double GetScalar(double strength, EasingFunction easingCurve, DateTime stopAt, int millis) {
        double fraction = (stopAt - DateTime.Now).TotalMilliseconds / millis;
        double easedValue = easingCurve.GetEasedValue((float)fraction) * strength;
        return easedValue;
        // return Math.Min(Math.Max(easedValue, 0), 1); // clamp it
    }
}