﻿using System;

namespace Celeste.Mod.CelesteButtplug;

public enum EasingFunction
{
    Linear,
    EaseInSine,
    EaseOutSine,
    EaseInOutSine,
    EaseInCubic,
    EaseOutCubic,
    EaseInOutCubic,
    EaseInQuint,
    EaseOutQuint,
    EaseInOutQuint,
    EaseInCirc,
    EaseOutCirc,
    EaseInOutCirc,
    EaseInElastic,
    EaseOutElastic,
    EaseInOutElastic,
    EaseInQuad,
    EaseOutQuad,
    EaseInOutQuad,
    EaseInQuart,
    EaseOutQuart,
    EaseInOutQuart,
    EaseInExpo,
    EaseOutExpo,
    EaseInOutExpo,
    EaseInBack,
    EaseOutBack,
    EaseInOutBack,
    EaseInBounce,
    EaseOutBounce,
    EaseInOutBounce,
}

public static class ExtensionMethods
{
    public static double GetEasedValue(this EasingFunction easingFunction, float x)
    {
        const double c1 = 1.70158;
        const double c2 = c1 * 1.525;
        const double c3 = c1 + 1;
        const double c4 = 2 * Math.PI / 3;
        const double c5 = 2 * Math.PI / 4.5;
        // From https://easings.net
        switch(easingFunction)
        {
            default:
            case EasingFunction.Linear:
                return x;
            case EasingFunction.EaseInSine:
                return 1 - Math.Cos(x * Math.PI / 2);
            case EasingFunction.EaseOutSine:
                return Math.Sin(x * Math.PI / 2);
            case EasingFunction.EaseInOutSine:
                return -(Math.Cos(Math.PI * x) - 1) / 2;
            case EasingFunction.EaseInCubic:
                return x * x * x;
            case EasingFunction.EaseOutCubic:
                return 1 - Math.Pow(1 - x, 3);
            case EasingFunction.EaseInOutCubic:
                return x < 0.5 ? 4 * x * x * x : 1 - Math.Pow(-2 * x + 2, 3) / 2;
            case EasingFunction.EaseInQuint:
                return x * x * x * x * x;
            case EasingFunction.EaseOutQuint:
                return 1 - Math.Pow(1 - x, 5);
            case EasingFunction.EaseInOutQuint:
                return x < 0.5 ? 16 * x * x * x * x * x : 1 - Math.Pow(-2 * x + 2, 5) / 2;
            case EasingFunction.EaseInCirc:
                return 1 - Math.Sqrt(1 - Math.Pow(x, 2));
            case EasingFunction.EaseOutCirc:
                return Math.Sqrt(1 - Math.Pow(x - 1, 2));
            case EasingFunction.EaseInOutCirc:
                return x < 0.5 ? (1 - Math.Sqrt(1 - Math.Pow(2 * x, 2))) / 2 : (Math.Sqrt(1 - Math.Pow(-2 * x + 2, 2)) + 1) / 2;
            case EasingFunction.EaseInElastic:
                if (x is 0 or 1) return x;
                return -Math.Pow(2, 10 * x - 10) * Math.Sin((x * 10 - 10.75) * c4);
            case EasingFunction.EaseOutElastic:
                if (x is 0 or 1) return x;
                return Math.Pow(2, -10 * x) * Math.Sin((x * 10 - 0.75) * c4) + 1;
            case EasingFunction.EaseInOutElastic:
                if (x is 0 or 1) return x;
                return x < 0.5 ? -(Math.Pow(2, 20 * x - 10) * Math.Sin((20 * x - 11.125) * c5)) / 2 : Math.Pow(2, -20 * x + 10) * Math.Sin((20 * x - 11.125) * c5) / 2 + 1;
            case EasingFunction.EaseInQuad:
                return x * x;
            case EasingFunction.EaseOutQuad:
                return 1 - (1 - x) * (1 - x);;
            case EasingFunction.EaseInOutQuad:
                return x < 0.5 ? 2 * x * x : 1 - Math.Pow(-2 * x + 2, 2) / 2;
            case EasingFunction.EaseInQuart:
                return x * x * x * x;
            case EasingFunction.EaseOutQuart:
                return 1 - Math.Pow(1 - x, 4);
            case EasingFunction.EaseInOutQuart:
                return x < 0.5 ? 8 * x * x * x * x : 1 - Math.Pow(-2 * x + 2, 4) / 2;
            case EasingFunction.EaseInExpo:
                return x is 0 ? x : Math.Pow(2, 10 * x - 10);
            case EasingFunction.EaseOutExpo:
                return x is 1 ? x : 1 - Math.Pow(2, -10 * x);
            case EasingFunction.EaseInOutExpo:
                if (x is 0 or 1) return x;
                return x < 0.5 ? Math.Pow(2, 20 * x - 10) / 2 : (2 - Math.Pow(2, -20 * x + 10)) / 2;
            case EasingFunction.EaseInBack:
                return c3 * x * x * x - c1 * x * x;
            case EasingFunction.EaseOutBack:
                return 1 + c3 * Math.Pow(x - 1, 3) + c1 * Math.Pow(x - 1, 2);
            case EasingFunction.EaseInOutBack:
                return x < 0.5 ? Math.Pow(2 * x, 2) * ((c2 + 1) * 2 * x - c2) / 2 : (Math.Pow(2 * x - 2, 2) * ((c2 + 1) * (x * 2 - 2) + c2) + 2) / 2;
            case EasingFunction.EaseInBounce:
                return 1 - EasingFunction.EaseOutBounce.GetEasedValue(1 - x);
            case EasingFunction.EaseOutBounce:
                const float n1 = 7.5625f;
                const float d1 = 2.75f;

                switch (x)
                {
                    case < 1 / d1:
                        return n1 * x * x;
                    case < 2 / d1:
                        return n1 * (x -= 1.5f / d1) * x + 0.75;
                    default:
                    {
                        if (x < 2.5 / d1) {
                            return n1 * (x -= 2.25f / d1) * x + 0.9375;
                        }
                        return n1 * (x -= 2.625f / d1) * x + 0.984375;
                    }
                }
            case EasingFunction.EaseInOutBounce:
                return x < 0.5
                    ? (1 - EasingFunction.EaseOutBounce.GetEasedValue(1 - 2 * x)) / 2
                    : (1 + EasingFunction.EaseOutBounce.GetEasedValue(2 * x - 1)) / 2;
        }
    }
}