local luaReflection = {}

local luanet = _G.luanet
local generalBindingFlags = luanet.enum(luanet.import_type("System.Reflection.BindingFlags"), 32 | 16 | 4 | 8)

local FakeAssembly = require("#Celeste.Mod.Helpers.FakeAssembly")
local Assembly = require("#System.Reflection.Assembly")
local Vector2 = require("#microsoft.xna.framework.vector2")
local Convert = require("#System.Convert")
local Array = require("#System.Array")
local SystemType = require("#System.Type")

local XNAAssembly = Assembly.GetAssembly(Vector2():GetType())
local SystemObjectType = SystemType.GetType("System.Object")
local SystemTypeType = SystemType.GetType("System.Type")

local function logMissingType(typeName)
    helpers.log("Couldn't find type " .. typeName)
    return nil
end

function luaReflection.getType(typeName)
    return FakeAssembly.GetFakeEntryAssembly():GetType(typeName, false, true) or XNAAssembly:GetType(typeName, false, true) or logMissingType(typeName)
end

function luaReflection.getTypes(types)
    local ret = Array.CreateInstance(SystemTypeType, #types)
    for i,v in ipairs(types) do
        ret[i - 1] = luaReflection.getType(v)
    end
    return ret
end

function luaReflection.luaTableToObjectArray(table)
    local ret = Array.CreateInstance(SystemObjectType, #table)

    for i,v in ipairs(table) do
        ret[i - 1] = v
    end

    return ret
end

function luaReflection.getField(typeName, fieldName)
    return luaReflection.getType(typeName):GetField(fieldName, generalBindingFlags)
end

---@param typeName string
---@param fieldName string
---@param obj csobject
function luaReflection.getFieldValue(typeName, fieldName, obj)
	return luaReflection.getField(typeName, fieldName):GetValue(obj)
end

function luaReflection.get(obj, fieldName)
    return obj:GetType():GetField(fieldName, generalBindingFlags):GetValue(obj)
end

function luaReflection.setField(obj, field, value)
    if (field.FieldType.Name == "Int32") then
        field:SetValue(obj, Convert.ToInt32(value))
    elseif (field.FieldType.Name == "Single") then
        field:SetValue(obj, Convert.ToSingle(value)) -- unfortunately, lua doesn't support floats, only doubles :( this will not work correctly iirc
    else
        field:SetValue(obj, value)
    end
end
---@param typeName string
---@param fieldName string
---@param obj csobject
---@param value any
function luaReflection.setFieldValue(typeName, fieldName, obj, value)
    local field = luaReflection.getField(typeName, fieldName)
    luaReflection.setField(obj, field, value)
end

function luaReflection.set(obj, fieldName, value)
    luaReflection.setField(obj, obj:GetType():GetField(fieldName, generalBindingFlags), value)
end

function luaReflection.call(obj, funcName, params)
    local m = obj:GetType():GetMethod(funcName, generalBindingFlags)

    -- Marshall lua tables to c# object[]
    if type(params) == "table" then
        params = luaReflection.luaTableToObjectArray(params)
    end

    m:Invoke(obj, params)
end

function luaReflection.callCoroutine(obj, funcName, params)
    m = obj:GetType():GetMethod(funcName, generalBindingFlags)

    -- Marshall lua tables to c# object[]
    if type(params) == "table" then
        params = luaReflection.luaTableToObjectArray(params)
    end

    return m:Invoke(obj, params)
end

function luaReflection.getMethod(obj, methodName, generic, methodParamTypes)
    local methods = obj:GetType():GetMethods(generalBindingFlags)
    for _, v in ipairs(methods) do
        if v.Name == methodName and v.IsGenericMethod == generic then
            if methodParamTypes then
                if v:GetParameters().Length == #methodParamTypes then
                    local params = v:GetParameters()
                    local shouldReturn = true
                    for i = 0, params.Length - 1, 1 do
                        if shouldReturn then
                            shouldReturn = params[i].ParameterType.Name == methodParamTypes[i+1]
                        end
                    end
                    if shouldReturn == true then return v end
                end
            else
                return v
            end
        end
    end
    log("Couldn't find method " .. methodName .. " for type " .. obj:GetType().Name)
    return nil
end

---@param methodName string
---@param obj csobject
---@param TtypeNameTable table
---@param parametersTable table
---@return any
function luaReflection.callGenericMethod(obj, methodName, methodParamTypes, TtypeNameTable, parametersTable)
    return luaReflection.getMethod(obj, methodName, true, methodParamTypes):MakeGenericMethod(luaReflection.getTypes(TtypeNameTable)):Invoke(obj, luaReflection.luaTableToObjectArray(parametersTable))
end

return luaReflection