﻿using System;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using FMOD.Studio;
using Microsoft.Xna.Framework;
using Monocle;

namespace Celeste.Mod.OneHourChallenge;

public class ChallengeModule : EverestModule
{
    public static ChallengeModule Instance { get; private set; }

    public override Type SettingsType => typeof(ChallengeModuleSettings);
    public static ChallengeModuleSettings Settings => (ChallengeModuleSettings)Instance._Settings;

    public override Type SaveDataType => typeof(ChallengeSaveData);
    public static ChallengeSaveData SaveData => (ChallengeSaveData)Instance._SaveData;

    // dictionary that stores the state of each file
    public readonly Dictionary<int, bool> HCFiles = [];

    private const int FramesPerSecond = 60;
    private const int FramesPerMinute = FramesPerSecond * 60;
    private const int FramesPerHour = FramesPerMinute * 60;

    public readonly Dictionary<int, int> IndexToTime = new() {
        { 0, 15 * FramesPerMinute }, // 15 minutes
        { 1, 30 * FramesPerMinute }, // 30 minutes
        { 2, 45 * FramesPerMinute }, // 45 minutes
        { 3, 1 * FramesPerHour }, // 1 hour
        { 4, 2 * FramesPerHour } // 2 hours
    };

    public readonly Dictionary<int, int> TimeToIndex = new() {
        { 15 * FramesPerMinute, 0 }, // 15 minutes
        { 30 * FramesPerMinute, 1 }, // 30 minutes
        { 45 * FramesPerMinute, 2 }, // 45 minutes
        { 1 * FramesPerHour, 3 }, // 1 hour
        { 2 * FramesPerHour, 4 } // 2 hours
    };

    // current file select slot (for OnHardcoreToggleSelected)
    private OuiFileSelectSlot currentSlot;

    // button to toggle hardcore mode
    private OuiFileSelectSlot.Button toggleButton;

    public ChallengeModule()
    {
        Instance = this;

    #if DEBUG
        // debug builds use verbose logging
        Logger.SetLogLevel(nameof(OneHourChallenge), LogLevel.Verbose);
    #else
        // release builds use info logging to reduce spam in log files
        Logger.SetLogLevel(nameof(OneHourChallenge), LogLevel.Info);
    #endif
    }
    public override void Load()
    {
        On.Celeste.SpeedrunTimerDisplay.DrawTime += SpeedrunTimerDisplay_DrawTime;
        On.Celeste.Level.Update += Level_Update;
        On.Celeste.OuiFileSelectSlot.Render += On_OuiFileSelectSlot_Render;
        On.Celeste.OuiFileSelectSlot.Select += On_OuiFileSelectSlot_Select;
        On.Celeste.OuiFileSelectSlot.CreateButtons += On_OuiFileSelectSlot_CreateButtons;
        On.Celeste.OuiFileSelectSlot.OnNewGameSelected += On_OuiFileSelectSlot_OnNewGameSelected;
        On.Celeste.OuiJournal.Render += On_OuiJournal_Render;
        On.Celeste.SaveData.TryDelete += On_SaveData_TryDelete;
    }

    public override void Unload()
    {
        On.Celeste.SpeedrunTimerDisplay.DrawTime -= SpeedrunTimerDisplay_DrawTime;
        On.Celeste.Level.Update -= Level_Update;
        On.Celeste.OuiFileSelectSlot.Render -= On_OuiFileSelectSlot_Render;
        On.Celeste.OuiFileSelectSlot.Select -= On_OuiFileSelectSlot_Select;
        On.Celeste.OuiFileSelectSlot.CreateButtons -= On_OuiFileSelectSlot_CreateButtons;
        On.Celeste.OuiFileSelectSlot.OnNewGameSelected -= On_OuiFileSelectSlot_OnNewGameSelected;
        On.Celeste.OuiJournal.Render -= On_OuiJournal_Render;
        On.Celeste.SaveData.TryDelete -= On_SaveData_TryDelete;
    }

    public override void CreateModMenuSection(TextMenu menu, bool inGame, EventInstance snapshot)
    {
        base.CreateModMenuSection(menu, inGame, snapshot);

        menu.Add(
            new TextMenu.Slider(
                Dialog.Clean("ONEHOURCHALLENGE_TIME"),
                i => IntToTime(IndexToTime.TryGetValue(i, out int time) ? time : Settings.StartTime),
                0, IndexToTime.Count - 1, TimeToIndex.GetValueOrDefault(Settings.StartTime, -1)
            ).Change(v => Settings.StartTime = IndexToTime[v]));
    }

    #region Hooks

    [SuppressMessage("ReSharper", "PossibleInvalidCastExceptionInForeachLoop",
        Justification = "Cast will always succeed")]
    private static void Level_Update(
        On.Celeste.Level.orig_Update orig,
        Level self)
    {
        orig(self);
        if (self.Completed || !SaveData.HourChallengeModeEnabled)
            return;

        if (SaveData.TimeRemaining >= 1)
        {
            SaveData.TimeRemaining -= 1;
            return;
        }

        // TIME'S UP
        if (self.Tracker.GetEntity<Player>() is not { } player)
            return;

        PlayerDeadBody body = player.Die(Vector2.Zero, true, false);
        body.DeathAction = () =>
        {
            int curScore = CalcScore(global::Celeste.SaveData.Instance);

            bool newBest = Settings.BestScore < curScore;
            if (newBest)
                Settings.BestScore = curScore;

            Engine.TimeRate = 1f;
            self.Session.InArea = false;
            Audio.BusStopAll("bus:/gameplay_sfx", immediate: true);
            Engine.Scene = new GameOverScreen(newBest);
        };

        // Freeze players
        foreach (Player player2 in self.Tracker.GetEntities<Player>())
        {
            player2.Speed = Vector2.Zero;
            player2.StateMachine.State = Player.StDummy;
        }
    }

    /// <summary>
    /// Resets a file's Hour Challenge Mode status when it is successfully deleted.
    /// </summary>
    /// <param name="orig">The original method.</param>
    /// <param name="slot">The ID of the file being deleted.</param>
    /// <returns>The result of the original method (whether the file was successfully deleted).</returns>
    private static bool On_SaveData_TryDelete(
        On.Celeste.SaveData.orig_TryDelete orig,
        int slot)
    {
        bool result = orig(slot);

        if (result)
            Instance.HCFiles[slot] = false;

        return result;
    }

    /// <summary>
    /// Draws the hardcore tab if that file is in hardcore mode.
    /// </summary>
    /// <param name="orig">The original method.</param>
    /// <param name="self">The file select slot we are drawing on.</param>
    private static void On_OuiFileSelectSlot_Render(
        On.Celeste.OuiFileSelectSlot.orig_Render orig,
        OuiFileSelectSlot self)
    {
        if (!Instance.Is1HCFile(self.FileSlot))
        {
            orig(self);
            return;
        }

        // mimic the original code for drawing tabs
        float highlightEase = self.highlightEase;
        float newGameFade = self.newgameFade;
        float scaleFactor = Ease.CubeInOut(highlightEase);
        bool selected = self.highlighted;

        Vector2 pos = selected ? self.Position + Vector2.UnitX * 700f * scaleFactor : self.Position;
        Vector2 vector = pos - new Vector2(scaleFactor * 360f, 0f); // ((Vector2.UnitX * scaleFactor * 360f));
        float scale = self.Exists ? 1f : newGameFade;

        if (!self.Corrupted && (newGameFade > 0f || self.Exists))
            MTN.FileSelect["1hctab"].DrawCentered(vector, Color.White * scale, 1f, Calc.ToRad(180));

        orig(self);
    }

    private static void On_OuiFileSelectSlot_Select(
        On.Celeste.OuiFileSelectSlot.orig_Select orig,
        OuiFileSelectSlot self,
        bool resetButtonIndex)
    {
        if (!self.Exists && resetButtonIndex)
        {
            Instance.HCFiles[self.FileSlot] = false;
            self.Portrait.Play("idle_normal");
        }
        orig(self, resetButtonIndex);
    }

    /// <summary>
    /// Adds the "1HourChallenge" button to file creation.
    /// </summary>
    /// <param name="orig">The original method.</param>
    /// <param name="self">The file select slot we are creating.</param>
    private static void On_OuiFileSelectSlot_CreateButtons(
        On.Celeste.OuiFileSelectSlot.orig_CreateButtons orig,
        OuiFileSelectSlot self)
    {
        orig(self);
        Instance.currentSlot = self;
        if (self.Exists)
            return;

        string dialogId = Instance.HCFiles[self.FileSlot]
            ? "FILE_HOURCHALLENGE_ON"
            : "FILE_HOURCHALLENGE_OFF";

        Instance.toggleButton = new OuiFileSelectSlot.Button {
            Label = Dialog.Clean(dialogId),
            Action = Instance.OnHCToggleSelected,
            Scale = 0.7f,
        };

        self.buttons.Add(Instance.toggleButton);
    }

    /// <summary>
    /// Writes the mod's SaveData to a newly created save file.
    /// </summary>
    /// <param name="orig">The original method.</param>
    /// <param name="self">The file select slot being created.</param>
    private static void On_OuiFileSelectSlot_OnNewGameSelected(
        On.Celeste.OuiFileSelectSlot.orig_OnNewGameSelected orig,
        OuiFileSelectSlot self)
    {
        orig(self);
        // the obsolescence warning mentions specifically overriding, calling is fine (probably)
#pragma warning disable CS0618 // Type or member is obsolete
        Instance.LoadSaveData(self.FileSlot);
        SaveData.HourChallengeModeEnabled = Instance.HCFiles[self.FileSlot];
        SaveData.TimeRemaining = Settings.StartTime;
        SaveData.StartTime = Settings.StartTime;
        Instance.SaveSaveData(self.FileSlot);
#pragma warning restore CS0618 // Type or member is obsolete
    }

    /// <summary>
    /// Shows the hardcore tab on the journal screen.
    /// </summary>
    /// <param name="orig">The original method.</param>
    /// <param name="self">The journal we are drawing on.</param>
    private static void On_OuiJournal_Render(
        On.Celeste.OuiJournal.orig_Render orig,
        OuiJournal self)
    {
        Vector2 vector = self.Position + new Vector2(128f, 120f);
        if (SaveData.HourChallengeModeEnabled)
            MTN.FileSelect["1hctab"].DrawCentered(
                vector + new Vector2(80f, 425f), Color.White, 1f, MathF.PI / 2);
        orig(self);
    }

    #endregion

    public static int CalcScore(SaveData data)
    {
        int score = 0;
        foreach (AreaStats areaStats in data.Areas_Safe)
        {
            for (int i = 0; i < areaStats.Modes.Length; i++)
            {
                if (areaStats.Modes[i] is null)
                    continue;

                // Hearts
                if (areaStats.Modes[i].HeartGem)
                    score += 10 * (i + 1);

                // Completed Levels
                score += areaStats.Modes[i].SingleRunCompleted ? 30 * (i + 1) : 0;
            }
            // Berries
            score += areaStats.TotalStrawberries * 5;

            // Cassettes
            score += areaStats.Cassette ? 10 : 0;
        }
        return score;
    }

    /// <summary>
    /// Reads a file slot's dictionary entry to determine whether it is in 1 Hour Challenge mode.
    /// If the entry is not present in the dictionary, it will add it.
    /// </summary>
    /// <param name="slot">The slot number.</param>
    /// <returns>True if the file is in 1 Hour Challenge mode, false otherwise.</returns>
    private bool Is1HCFile(int slot)
    {
        // attempt to get from dictionary
        if (HCFiles.TryGetValue(slot, out bool value))
            return value;

        // default to false if the save file doesn't exist at all
        if (!UserIO.Exists(global::Celeste.SaveData.GetFilename(slot)))
            return false;

        // attempt to load from save file
        try
        {
            Logger.Info(nameof(OneHourChallenge), $"Getting save data from save slot {slot}.");

            // the obsolescence warning mentions specifically overriding, calling is fine (probably)
#pragma warning disable CS0618 // Type or member is obsolete
            LoadSaveData(slot);
#pragma warning restore CS0618 // Type or member is obsolete

            HCFiles[slot] = SaveData.HourChallengeModeEnabled;
            return HCFiles[slot];
        }
        catch
        {
            // default to false if that fails somehow
            Logger.Warn(nameof(OneHourChallenge), $"Could not get save data from save slot {slot}.");
            HCFiles[slot] = false;
            return false;
        }
    }

    private void OnHCToggleSelected()
    {
        int index = currentSlot.FileSlot;
        if (!Is1HCFile(index))
        {
            HCFiles[index] = true;
            toggleButton.Label = Dialog.Clean("FILE_HOURCHALLENGE_ON");
            Audio.Play("event:/ui/main/button_toggle_on");
        }
        else
        {
            HCFiles[index] = false;
            toggleButton.Label = Dialog.Clean("FILE_HOURCHALLENGE_OFF");
            Audio.Play("event:/ui/main/button_toggle_off");
        }
    }

    public static string IntToTime(int time)
    {
        int minutes = time / FramesPerMinute % 60;
        int seconds = time / FramesPerSecond % 60;
        int hours = time / FramesPerHour;

        string hh = hours.ToString();
        string mm = minutes.ToString("00");
        string ss = seconds.ToString("00");

        return time / 216000 > 0
            ? $"{hh}:{mm}.{ss}"
            : $"{mm}.{ss}";
    }

    private static void SpeedrunTimerDisplay_DrawTime(
        On.Celeste.SpeedrunTimerDisplay.orig_DrawTime orig,
        Vector2 position,
        string timeString,
        float scale,
        bool valid,
        bool finished,
        bool bestTime,
        float alpha)
    {
        if (!SaveData.HourChallengeModeEnabled)
        {
            orig(position, timeString, scale, valid, finished, bestTime, alpha);
            return;
        }

        if (timeString.StartsWith("Score: "))
        {
            orig(position, timeString, scale, valid, finished, bestTime, alpha);
            return;
        }

        if (timeString == "Time's Up!")
        {
            orig(position, timeString, scale, valid, finished, bestTime, alpha);
            return;
        }

        timeString = IntToTime(SaveData.TimeRemaining);
        int score = CalcScore(global::Celeste.SaveData.Instance);
        string scoreString = $"Score: {score}";
        Vector2 scorePos = SaveData.TimeRemaining > 0
            ? position + Vector2.UnitY * 50f
            : new Vector2(
                (Celeste.TargetWidth - SpeedrunTimerDisplay.numberWidth * scoreString.Length) / 2,
                Celeste.TargetHeight / 2);

        SpeedrunTimerDisplay.DrawTime(
            scorePos, scoreString, scale, valid: true, finished: false, bestTime: score > Settings.BestScore, alpha);

        // if (SaveData.TimeRemaining < 1)
        //     SpeedrunTimerDisplay.DrawTime(
        //         scorePos - Vector2.UnitY * 50f, "Time's Up!", scale, valid: true, finished: false, bestScore: score > Settings.BestScore, alpha);

        orig(position, timeString, scale, valid, finished, bestTime, alpha);
    }
}
