-- Paste Script (With Validation and Improved Handling)
local utils = require("utils")
local map_item_utils = require("map_item_utils")
local loadedState = require("loaded_state")
local roomStruct = require("structs.room")
local versionss = 7
local offset = 5000
local script = {
    name = "pasteAllRooms",
    displayName = "Paste All Rooms",
    tooltip = "Paste all rooms from the clipboard into the current map, creating new rooms with unique names"
}

-- Custom split function
local function split(inputstr, sep)
    if sep == nil then
        sep = "%s"
    end
    local t = {}
    for str in string.gmatch(inputstr, "([^"..sep.."]+)") do
        table.insert(t, str)
    end
    return t
end

function script.run(_, args)
    local map = loadedState.map
    if map then
        -- Get clipboard data
        local clipboardText = love.system.getClipboardText()
        if not clipboardText or clipboardText == "" then
            print("No clipboard data found")
            return  -- Exit if no clipboard data found
        end

        -- Split the clipboard text into individual room data blocks based on "+"
        local roomDataList = split(clipboardText, "+")

        -- Check if there are room data blocks to process
        if #roomDataList == 0 then
            print("No valid room data found in the clipboard")
            return
        end

        -- Iterate through each room's serialized data and deserialize it
        for _, roomText in ipairs(roomDataList) do
            -- Skip empty or invalid room data
            if roomText == "" then
                print("Skipping empty room data")
                goto continue
            end

            local success, roomData = utils.unserialize(roomText)  -- Deserialize room with structure and contents
            if success and type(roomData) == "table" then  -- Ensure roomData is a table
                if roomData  then
                    local newRoom = roomStruct.decode(roomData)  -- Decode room structure and contents

                    if newRoom then
                        -- Generate a new unique name for the room
                        local duplicateCount = 1
                        local name = newRoom.name
                        while loadedState.getRoomByName(name) do
                            name = string.format(newRoom.name .. " (%d)", duplicateCount)
                            duplicateCount = duplicateCount + 1
                        end
                        newRoom.name = name  -- Update the room's name to make it unique

                        -- Add the new room to the map
                        if duplicateCount < versionss + 1 then
                            newRoom.x = newRoom.x + offset*duplicateCount - offset
                            newRoom.y = newRoom.y + offset*duplicateCount - offset
                            map_item_utils.addRoom(map, newRoom)
                         
                            print("Room added with name:", newRoom.name)
                        end
                    else
                        print("Failed to decode room:", roomData)
                    end
                else
                    print("Invalid room data structure (missing 'name' or 'tiles'):", roomData)
                end
            else
                print("Failed to unserialize room data:", roomText)
            end

            ::continue::  -- Continue to the next iteration
        end

        -- Refresh the map to render the new rooms
        if loadedState.refreshMap then
            loadedState.refreshMap()  -- Trigger map refresh to render the new rooms
        end
    else
        print("No map loaded")
    end
end

return script