﻿using System;
using System.Collections;
using Celeste.Mod.Entities;
using Microsoft.Xna.Framework;
using Monocle;

namespace Celeste.Mod.xoliSSC {
    public class xoliSSCModule : EverestModule {
        public static xoliSSCModule Instance { get; private set; }

        public override Type SettingsType => typeof(xoliSSCSettings);
        public static xoliSSCSettings Settings => (xoliSSCSettings) Instance._Settings;

        public override Type SessionType => typeof(xoliSSCSession);
        public static xoliSSCSession Session => (xoliSSCSession) Instance._Session;

        public xoliSSCModule() {
            Instance = this;
        }

        public override void Load() {
            // TODO: apply any hooks that should always be active
        }

        public override void Unload() {
            // TODO: unapply any hooks applied in Load()
        }
    }

    public class FactoryActivator : Component
    {
        public bool Activated => ActivationCount > 0;
        public int ActivationCount { get; private set; } = 0;
        public bool StartOn { get; set; }
        public bool IsOn
        {
            get
            {
                return Activated != StartOn;
            }
        }
        public bool StateIsLocked { get; set; } = false;

        public string ActivationId;

        public Action OnTurnOn = null;
        public Action OnTurnOff = null;
        public Action OnStartOn = null;
        public Action OnStartOff = null;

        public FactoryActivator() : base(true, true)
        {
        }

        public void ForceActivate()
        {
            bool wasOn = IsOn;
            StartOn = false;
            ActivationCount = 1;
            HandleOnOff(wasOn);
            StateIsLocked = true;
        }

        public void ForceDeactivate()
        {
            bool wasOn = IsOn;
            StartOn = false;
            ActivationCount = 0;
            HandleOnOff(wasOn);
            StateIsLocked = true;
        }

        public void Activate(bool lockState = true)
        {
            if (!StateIsLocked)
            {
                bool wasOn = IsOn;
                ActivationCount++;
                HandleOnOff(wasOn);
                StateIsLocked = lockState;
            }
        }

        public void Deactivate(bool lockState = true)
        {
            if (!StateIsLocked)
            {
                bool wasOn = IsOn;
                ActivationCount--;
                HandleOnOff(wasOn);
                StateIsLocked = lockState;
            }
        }

        private void HandleOnOff(bool wasOn)
        {
            if (IsOn != wasOn)
            {
                if (IsOn)
                {
                    OnTurnOn?.Invoke();
                }
                else
                {
                    OnTurnOff?.Invoke();
                }
            }
        }

        public void HandleStartup(Scene scene)
        {
            if (ActivationId == null)
            {
                ActivationCount = 0;
            }
            else
            {
                Level level = scene as Level;
                ActivationCount += level.Session.GetFlag($"FactoryActivation:{ActivationId}") ? 1 : 0;
            }
            if (IsOn)
            {
                OnStartOn?.Invoke();
            }
            else
            {
                OnStartOff?.Invoke();
            }
        }
    }

    [CustomEntity("xoliSSC/FunkyBoombox")]
    public class BoomBoxZip : Solid {
        public ParticleType P_Steam = ParticleTypes.Steam;
        public ParticleType P_SteamAngry = new ParticleType(ParticleTypes.Steam)
        {
            LifeMin = 1f,
            LifeMax = 2f,
            SpeedMin = 12f,
            SpeedMax = 24f
        };

        
        private class ZipMoverPathRenderer : Entity {
            

            private static readonly Color ropeColor = Calc.HexToColor("4d3c22");

            private static readonly Color ropeLightColor = Calc.HexToColor("766c49");

            public BoomBoxZip zipMover;

            private MTexture cog;

            private Vector2 from;

            private Vector2 to;

            private Vector2 sparkAdd;

            private float sparkDirFromA;

            private float sparkDirFromB;

            private float sparkDirToA;

            private float sparkDirToB;

            public ZipMoverPathRenderer(BoomBoxZip zipMover) {
                base.Depth = 5000;
                this.zipMover = zipMover;
                from = this.zipMover.start + new Vector2(this.zipMover.Width / 2f, this.zipMover.Height / 2f);
                to = this.zipMover.target + new Vector2(this.zipMover.Width / 2f, this.zipMover.Height / 2f);
                sparkAdd = (from - to).SafeNormalize(5f).Perpendicular();
                float num = (from - to).Angle();
                sparkDirFromA = num + (float)Math.PI / 8f;
                sparkDirFromB = num - (float)Math.PI / 8f;
                sparkDirToA = num + (float)Math.PI - (float)Math.PI / 8f;
                sparkDirToB = num + (float)Math.PI + (float)Math.PI / 8f;
                cog = GFX.Game["objects/zipmover/cog"];
            }

            public void CreateSparks() {
                SceneAs<Level>().ParticlesBG.Emit(ZipMover.P_Sparks, from + sparkAdd + Calc.Random.Range(-Vector2.One, Vector2.One), sparkDirFromA);
                SceneAs<Level>().ParticlesBG.Emit(ZipMover.P_Sparks, from - sparkAdd + Calc.Random.Range(-Vector2.One, Vector2.One), sparkDirFromB);
                SceneAs<Level>().ParticlesBG.Emit(ZipMover.P_Sparks, to + sparkAdd + Calc.Random.Range(-Vector2.One, Vector2.One), sparkDirToA);
                SceneAs<Level>().ParticlesBG.Emit(ZipMover.P_Sparks, to - sparkAdd + Calc.Random.Range(-Vector2.One, Vector2.One), sparkDirToB);
            }

            public override void Render() {
                DrawCogs(Vector2.UnitY, Color.Black);
                DrawCogs(Vector2.Zero);
                Draw.Rect(new Rectangle((int)(zipMover.X + zipMover.Shake.X - 1f), (int)(zipMover.Y + zipMover.Shake.Y - 1f), (int)zipMover.Width + 2, (int)zipMover.Height + 2), Color.Black);
            }

            private void DrawCogs(Vector2 offset, Color? colorOverride = null) {
                Vector2 vector = (to - from).SafeNormalize();
                Vector2 value = vector.Perpendicular() * 3f;
                Vector2 value2 = -vector.Perpendicular() * 4f;
                float rotation = zipMover.percent * (float)Math.PI * 2f;
                Draw.Line(from + value + offset, to + value + offset, colorOverride.HasValue ? colorOverride.Value : ropeColor);
                Draw.Line(from + value2 + offset, to + value2 + offset, colorOverride.HasValue ? colorOverride.Value : ropeColor);
                for (float num = 4f - zipMover.percent * (float)Math.PI * 8f % 4f; num < (to - from).Length(); num += 4f) {
                    Vector2 value3 = from + value + vector.Perpendicular() + vector * num;
                    Vector2 value4 = to + value2 - vector * num;
                    Draw.Line(value3 + offset, value3 + vector * 2f + offset, colorOverride.HasValue ? colorOverride.Value : ropeLightColor);
                    Draw.Line(value4 + offset, value4 - vector * 2f + offset, colorOverride.HasValue ? colorOverride.Value : ropeLightColor);
                }

                cog.DrawCentered(from + offset, colorOverride.HasValue ? colorOverride.Value : Color.White, 1f, rotation);
                cog.DrawCentered(to + offset, colorOverride.HasValue ? colorOverride.Value : Color.White, 1f, rotation);
            }
        }

        private class BoomCollider : Entity
        {
            public BoomCollider(Vector2 position) : base(position)
            {
                Collider = new Circle(40f, 0, 0);
            }
        }

        private const float _angryResetTime = 2f;
        private const float _angryShootTime = 0.5f;
        private const float _idlePuffTime = 0.5f;
        private const float _activePuffTime = 0.1f;
        private const float _angryPuffTime = 0.02f;

        private readonly float _initialDelay;
        private readonly Sprite _sprite;
        private readonly Sprite _boomSprite;
        private readonly SoundSource _sfx;
        private readonly float _startupTime = 1.5f;
        private float _angryResetTimer = 0f;
        private float _angryShootTimer = _angryShootTime;
        private bool _angryMode = false;
        private bool _canGetAngry = false;
        private Coroutine _sequence;
        private bool _steamAnger = false;
        private Player playerRef;
        public float percent;
        public Vector2 start, target;
        private ZipMoverPathRenderer pathRenderer;

        public FactoryActivator Activator { get; private set; }

        public BoomBoxZip(EntityData data, Vector2 offset) : this(data.Position + offset, data.Float("initialDelay", 0f), data.Bool("startActive", false), data.Bool("ActivateOnZipEnd", false), data.Nodes[0] + offset)
        {
        }

        public BoomBoxZip(Vector2 position, float initialDelay, bool startActive, bool activeAtEndOfZip, Vector2 target) : base(position, 24, 24, false)
        {

            _initialDelay = initialDelay;

            Add(_sprite = new Sprite(GFX.Game, "objects/FactoryHelper/boomBox/"));
            _sprite.Add("idle", "idle", 0.2f, "idle");
            _sprite.Add("activating", "activating", 0.2f, "activating");
            _sprite.Add("active", "active", 0.15f, "active");
            _sprite.Add("angry", "angry", 0.05f, "angry");
            _sprite.Add("resetting", "resetting", 0.15f, "active");

            Add(_boomSprite = new Sprite(GFX.Game, "objects/FactoryHelper/boomBox/"));
            _boomSprite.Add("boom", "boom", 0.04f);
            _boomSprite.Color = Color.White * 0.5f;
            _boomSprite.Visible = false;
            _boomSprite.CenterOrigin();
            _boomSprite.Position = new Vector2(Width / 2, Height / 2);
            Add(_sfx = new SoundSource());
            _sfx.Position = new Vector2(Width / 2, Height / 2);
            Add(new LightOcclude(0.2f));
            if(activeAtEndOfZip) Add(new Coroutine(Sequence2()));
            else {
                Add(new Coroutine(Sequence1()));
                
                Add(Activator = new FactoryActivator());
                Activator.StartOn = startActive;
                Activator.ActivationId = null;
                Activator.OnStartOn = OnStartOn;
                Activator.OnStartOff = OnStartOff;
                Activator.OnTurnOn = OnTurnOn;
                Activator.OnTurnOff = OnTurnOff;
            }
        }

        public override void Added(Scene scene)
        {
            base.Added(scene);
            scene.Add(pathRenderer = new ZipMoverPathRenderer(this));
        }

        public override void Update()
        {
            base.Update();
            if (!_sfx.Playing)
            {
                _sfx.Play("event:/env/local/09_core/conveyor_idle");
            }

            if(!_steamAnger && _canGetAngry)
            {
                HandleAngryMode();
            }
            if(!_canGetAngry && Scene.OnInterval(_idlePuffTime))
            {
                SceneAs<Level>().ParticlesFG.Emit(P_Steam, 1, Center, new Vector2(8f, 8f));
            }
            else if (_canGetAngry && !_angryMode && Scene.OnInterval(_activePuffTime))
            {
                SceneAs<Level>().ParticlesFG.Emit(P_Steam, 1, Center, new Vector2(8f, 8f));
            }
            else if ((_angryMode || _steamAnger) && Scene.OnInterval(_angryPuffTime))
            {
                SceneAs<Level>().ParticlesFG.Emit(P_SteamAngry, 3, Center, new Vector2(8f, 8f), direction:Calc.Random.NextAngle());
            }

            if (_boomSprite.Visible && !_boomSprite.Active)
            {
                _boomSprite.Visible = false;
            }
        }

        private IEnumerator Sequence1() {
            
            start = Position;
            while (true) {
                if (!HasPlayerRider()) {
                    yield return null;
                    continue;
                }
                Input.Rumble(RumbleStrength.Medium, RumbleLength.Short);
                StartShaking(0.1f);
                yield return 0.1f;
                StopPlayerRunIntoAnimation = false;
                float at2 = 0f;
                while (at2 < 1f) {
                    yield return null;
                    at2 = Calc.Approach(at2, 1f, 2f * Engine.DeltaTime);
                    percent = Ease.SineIn(at2);
                    Vector2 vector = Vector2.Lerp(start, target, percent);
                    ScrapeParticlesCheck(vector);
                    if (Scene.OnInterval(0.1f)) {
                        pathRenderer.CreateSparks();
                    }

                    MoveTo(vector);
                }

                StartShaking(0.2f);
                Input.Rumble(RumbleStrength.Strong, RumbleLength.Medium);
                SceneAs<Level>().Shake();
                StopPlayerRunIntoAnimation = true;
                yield return 0.5f;
                StopPlayerRunIntoAnimation = false;
                at2 = 0f;
                while (at2 < 1f) {
                    yield return null;
                    at2 = Calc.Approach(at2, 1f, 0.5f * Engine.DeltaTime);
                    percent = 1f - Ease.SineIn(at2);
                    Vector2 position = Vector2.Lerp(target, start, Ease.SineIn(at2));
                    MoveTo(position);
                }

                StopPlayerRunIntoAnimation = true;
                StartShaking(0.2f);
                yield return 0.5f;
            }
        }
        private IEnumerator Sequence2() {
            yield break;
        }

        
        private void ScrapeParticlesCheck(Vector2 to) {
            if (!base.Scene.OnInterval(0.03f)) {
                return;
            }

            bool flag = to.Y != base.ExactPosition.Y;
            bool flag2 = to.X != base.ExactPosition.X;
            if (flag && !flag2) {
                int num = Math.Sign(to.Y - base.ExactPosition.Y);
                Vector2 value = (num != 1) ? base.TopLeft : base.BottomLeft;
                int num2 = 4;
                if (num == 1) {
                    num2 = Math.Min((int)base.Height - 12, 20);
                }

                int num3 = (int)base.Height;
                if (num == -1) {
                    num3 = Math.Max(16, (int)base.Height - 16);
                }

                if (base.Scene.CollideCheck<Solid>(value + new Vector2(-2f, num * -2))) {
                    for (int i = num2; i < num3; i += 8) {
                        SceneAs<Level>().ParticlesFG.Emit(ZipMover.P_Scrape, base.TopLeft + new Vector2(0f, (float)i + (float)num * 2f), (num == 1) ? (-(float)Math.PI / 4f) : ((float)Math.PI / 4f));
                    }
                }

                if (base.Scene.CollideCheck<Solid>(value + new Vector2(base.Width + 2f, num * -2))) {
                    for (int j = num2; j < num3; j += 8) {
                        SceneAs<Level>().ParticlesFG.Emit(ZipMover.P_Scrape, base.TopRight + new Vector2(-1f, (float)j + (float)num * 2f), (num == 1) ? ((float)Math.PI * -3f / 4f) : ((float)Math.PI * 3f / 4f));
                    }
                }
            } else {
                if (!flag2 || flag) {
                    return;
                }

                int num4 = Math.Sign(to.X - base.ExactPosition.X);
                Vector2 value2 = (num4 != 1) ? base.TopLeft : base.TopRight;
                int num5 = 4;
                if (num4 == 1) {
                    num5 = Math.Min((int)base.Width - 12, 20);
                }

                int num6 = (int)base.Width;
                if (num4 == -1) {
                    num6 = Math.Max(16, (int)base.Width - 16);
                }

                if (base.Scene.CollideCheck<Solid>(value2 + new Vector2(num4 * -2, -2f))) {
                    for (int k = num5; k < num6; k += 8) {
                        SceneAs<Level>().ParticlesFG.Emit(ZipMover.P_Scrape, base.TopLeft + new Vector2((float)k + (float)num4 * 2f, -1f), (num4 == 1) ? ((float)Math.PI * 3f / 4f) : ((float)Math.PI / 4f));
                    }
                }

                if (base.Scene.CollideCheck<Solid>(value2 + new Vector2(num4 * -2, base.Height + 2f))) {
                    for (int l = num5; l < num6; l += 8) {
                        SceneAs<Level>().ParticlesFG.Emit(ZipMover.P_Scrape, base.BottomLeft + new Vector2((float)l + (float)num4 * 2f, 0f), (num4 == 1) ? ((float)Math.PI * -3f / 4f) : (-(float)Math.PI / 4f));
                    }
                }
            }
        }
        private void OnStartOn()
        {
            _sprite.Play("active", true);
            _canGetAngry = true;
        }

        private void OnStartOff()
        {
            _sprite.Play("idle", true);
            _canGetAngry = false;
        }

        private void OnTurnOn()
        {
            ResetTimers();
            if (_sequence != null)
            {
                Remove(_sequence);
            }
            Add(_sequence = new Coroutine(StartupSequence()));
        }

        private void OnTurnOff()
        {
            ResetTimers();
            if (_sequence != null)
            {
                Remove(_sequence);
            }
            Add(_sequence = new Coroutine(WindDownSequence()));
        }

        private void ResetTimers()
        {
            _angryResetTimer = 0f;
            _angryShootTimer = _angryShootTime;
        }
        private IEnumerator StartupSequence()
        {
            _canGetAngry = false;
            _sprite.Play("activating", true);
            yield return _initialDelay;
            yield return _startupTime;
            _sprite.Play("resetting", true);
            _canGetAngry = true;
        }

        private IEnumerator WindDownSequence()
        {
            yield return _initialDelay;
            _canGetAngry = false;
            _sprite.Play("activating", true);
            yield return _startupTime;
            _sprite.Play("idle", true);
        }

        

        private void HandleAngryMode()
        {
            CheckForAngryMode();
            if (_angryMode)
            {
                if (_angryShootTimer > 0)
                {
                    _angryShootTimer -= Engine.DeltaTime;
                }
                else
                {
                    Explode();
                    ResetAngryMode();
                }
            }
            HandleAngryModeResetting();
        }

        private void CheckForAngryMode()
        {
            if (_angryResetTimer <= 0f && !_angryMode && HasPlayerRider())
            {
                _angryMode = true;
                _sprite.Play("angry", true);
            }
        }

        private void HandleAngryModeResetting()
        {
            if (_angryResetTimer > 0f)
            {
                _angryResetTimer -= Engine.DeltaTime;
                if (_angryResetTimer <= 0f)
                {
                    _sprite.Play("resetting", true);
                }
            }
        }

        private void ResetAngryMode()
        {
            _angryMode = false;
            _angryResetTimer = _angryResetTime;
            _angryShootTimer = _angryShootTime;
            _sprite.Play("activating", true);
        }
        private void Explode()
        {
            Audio.Play("event:/new_content/game/10_farewell/puffer_splode", Position);
            _boomSprite.Play("boom");
            _boomSprite.Visible = true;
            (Scene as Level).Displacement.AddBurst(Center, 0.35f, 4f, 64f, 0.5f);
            Player player = Scene.Tracker.GetEntity<Player>();
            Collidable = false;
            Vector2 temp = Position + new Vector2(Width / 2, Height / 2);
            if (player != null && new Circle(40, temp.X, temp.Y).Collide(player) && !Scene.CollideCheck<Solid>(player.Center, Center))
            {
                if (player.Bottom < Top && player.Top > Bottom)
                {
                    player.ExplodeLaunch(Center, false, true);
                }
                else
                {
                    player.ExplodeLaunch(Center, false, false);
                }
            }
            Collidable = true;
        }
    }
}