module CustomClutter

using ..Ahorn, Maple
using Random

@mapdef Entity "CustomClutterHelper/customClutterBlock" CustomClutterBlock(x::Integer, y::Integer, width::Integer=8, height::Integer=8, clutterFolder::String="objects/customclutter/", color::String="exampleclutter", surfaceSoundIndex::Integer=8)
@mapdef Entity "CustomClutterHelper/customClutterSwitch" CustomClutterSwitch(x::Integer, y::Integer, clutterFolder::String="objects/customclutter/", type::String="exampleclutter", absorbCutsceneSound::String="event:/game/03_resort/clutterswitch_books", switchSprite::String="clutterSwitch", progressMusic::Bool=true, lightingAlphaAdd::Number=0.05, temporaryLightingAlpha::Number=0.05, doTemporaryLightingChanges::Bool=true)

const placements = Ahorn.PlacementDict(
    "Custom Clutter Block (Custom Clutter)" => Ahorn.EntityPlacement(
        CustomClutterBlock,
        "rectangle",
        Dict{String, Any}(
            "clutterFolder" => "objects/customclutter/",
            "color" => "exampleclutter",
            "surfaceSoundIndex" => 8
        )
    ),
    "Custom Clutter Switch (Custom Clutter)" => Ahorn.EntityPlacement(
        CustomClutterSwitch,
        "point",
        Dict{String, Any}(
            "clutterFolder" => "objects/customclutter/",
            "type" => "exampleclutter",
            "absorbCutsceneSound" => "event:/game/03_resort/clutterswitch_books",
            "switchSprite" => "clutterSwitch"
        )
    )
)

textureName(entity::CustomClutterBlock, i::Integer) = get(entity.data, "clutterFolder", "objects/customclutter/") * lowercase(get(entity.data, "color", "exampleclutter")) * "_$(lpad(i, 2, "0"))"

function getTextures(entity::CustomClutterBlock)
    i = 0
    res = Ahorn.Sprite[]

    atlas = Ahorn.getAtlas("Gameplay")
    sprite = Ahorn.getSprite(textureName(entity, i), "Gameplay")
    while sprite.realWidth != 0 && sprite.realHeight != 0 && sprite.surface !== Ahorn.Assets.missingImage
        push!(res, sprite)
        i += 1
        sprite = Ahorn.getSprite(textureName(entity, i), "Gameplay")
    end

    return res
end

function getEntityRng(entity::CustomClutterBlock, dr::Ahorn.DrawableRoom)
    x = floor(Int, get(entity.data, "x", 0) / 8)
    y = floor(Int, get(entity.data, "y", 0) / 8)

    width = floor(Int, get(entity.data, "width", 32) / 8)
    height = floor(Int, get(entity.data, "height", 32) / 8)

    randStates = dr.bgTileStates.rands
    rands = get(randStates, (y:y + height - 1, x:x + width - 1), 1)

    # Rands are stored as 1:4, we need 0:3 here
    seed = abs(foldl((n, m) -> n << 2 | m, rands .- 1))

    return MersenneTwister(seed)
end

function renderCustomClutterBlock(ctx::Ahorn.Cairo.CairoContext, entity::CustomClutterBlock, dr::Ahorn.DrawableRoom)
    entityX = Int(get(entity.data, "x", 0))
    entityY = Int(get(entity.data, "y", 0))

    width = Int(get(entity.data, "width", 32))
    height = Int(get(entity.data, "height", 32))

    tw = ceil(Int, width / 8)
    th = ceil(Int, height / 8)

    needsDrawing = fill(true, (th, tw))
    rng = getEntityRng(entity, dr)
    textures = getTextures(entity)

    for y in 1:th, x in 1:tw
        if needsDrawing[y, x]
            choices = copy(textures)
            shuffle!(rng, choices)

            for choice in choices
                w, h = floor(Int, choice.width / 8), floor(Int, choice.height / 8)
                if all(get(needsDrawing, (y:y + h - 1, x:x + w - 1), false))
                    needsDrawing[y:y + h - 1, x:x + w - 1] .= false
                    Ahorn.drawImage(ctx, choice, entityX + x * 8 - 8, entityY + y * 8 - 8)

                    break
                end
            end
        end
    end
end


function Ahorn.selection(entity::CustomClutterSwitch)
    x, y = Ahorn.position(entity)

    return Ahorn.Rectangle(x - 4, y - 2, 40, 18)
end

function Ahorn.renderAbs(ctx::Ahorn.Cairo.CairoContext, entity::CustomClutterSwitch, room::Maple.Room)
    x, y = Ahorn.position(entity)
    
    clutterFolder = get(entity.data, "clutterFolder", "objects/customclutter/")
    variant = lowercase(get(entity.data, "type", "exampleclutter"))
    sprite = Ahorn.getSprite(clutterFolder * "icon_$variant", "Gameplay")

    Ahorn.drawImage(ctx, "objects/resortclutter/clutter_button00", x - 4, y - 2)
    Ahorn.drawImage(ctx, sprite, x + (32 - sprite.width) / 2, y + (16 - sprite.height) / 2)
end

Ahorn.minimumSize(entity::CustomClutterBlock) = 8, 8
Ahorn.resizable(entity::CustomClutterBlock) = true, true

Ahorn.selection(entity::CustomClutterBlock) = Ahorn.getEntityRectangle(entity)

function Ahorn.renderAbs(ctx::Ahorn.Cairo.CairoContext, entity::CustomClutterBlock, room::Maple.Room)
    dr = Ahorn.getDrawableRoom(Ahorn.loadedState.map, room)

    renderCustomClutterBlock(ctx, entity, dr)
end

end
