﻿using Celeste.Mod.Entities;
using Celeste.Mod.ScuffedHelper.Noise;
using Microsoft.Xna.Framework;
using Monocle;
using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Celeste.Mod.ScuffedHelper {
	//code copied from floatyspaceblock

    [CustomEntity("ScuffedHelper/PushPuzzleBlock")]
    [Tracked]
    class PushPuzzleBlock : Solid{

		private TileGrid tiles;
		private char tileType;

		public int Index;

        public Vector2 StartPos;
		public Vector2 TargetPos;
		public int moveDistance;

		private Vector2 dashDirection;

		private PushPuzzleBlock master;

		private bool awake;

		public List<PushPuzzleBlock> Group;

		//Platform, Offset of platform
		public List<KeyValuePair<PushPuzzleBlock, Vector2>> Moves;
		public Vector2 MasterOffset;

		public Point GroupBoundsMin;
		public Point GroupBoundsMax;

		public Color BlockColour;

		public bool HasGroup { get; private set; }
		public bool MasterOfGroup { get; private set; }

		public PushPuzzleBlock(EntityData data, Vector2 offset) : base(data.Position + offset, data.Width, data.Height, safe: true) {
            StartPos = data.Position + offset;
			TargetPos = StartPos;
			Index = data.Int("index", 0);

			moveDistance = data.Int("moveDistance", 32);

			tileType = data.Char("tiletype", 'P');
			Logger.Log("Scuffed Helper", $"{tileType}");

			//BlockColour = new Color(data.Int("r", 36), data.Int("g", 8), data.Int("b", 78), (int) (data.Float("a", 0.8f) * 255));
			BlockColour = new Color(data.Int("r", 36), data.Int("g", 8), data.Int("b", 78));
			BlockColour *= data.Float("a", 0.8f);
			Logger.Log("Scuffed Helper", $"{BlockColour.A}");
			//Logger.Log("Scuffed Helper", "Got here");
		}

        public override void Awake(Scene scene) {
            base.Awake(scene);
			awake = true;
			if (!HasGroup) {
				MasterOfGroup = true;
				master = this;
				MasterOffset = Vector2.Zero;
				Moves = new List<KeyValuePair<PushPuzzleBlock, Vector2>>();
				Group = new List<PushPuzzleBlock>();
				GroupBoundsMin = new Point((int)base.X, (int)base.Y);
				GroupBoundsMax = new Point((int)base.Right, (int)base.Bottom);
				AddToGroupAndFindChildren(this);

				//what does this do
				Rectangle rectangle = new Rectangle(GroupBoundsMin.X / 8, GroupBoundsMin.Y / 8, (GroupBoundsMax.X - GroupBoundsMin.X) / 8 + 1, (GroupBoundsMax.Y - GroupBoundsMin.Y) / 8 + 1);
				
				//I assume this is the tile map
				VirtualMap<char> virtualMap = new VirtualMap<char>(rectangle.Width, rectangle.Height, '0');

				//what does this do
				//I think this makes every tile in the group

				foreach (PushPuzzleBlock item in Group) {
					int num = (int)(item.X / 8f) - rectangle.X;
					int num2 = (int)(item.Y / 8f) - rectangle.Y;
					int num3 = (int)(item.Width / 8f);
					int num4 = (int)(item.Height / 8f);
					for (int i = num; i < num + num3; i++) {
						for (int j = num2; j < num2 + num4; j++) {
							virtualMap[i, j] = tileType;
						}
					}
				}	
				

				tiles = GFX.FGAutotiler.GenerateMap(virtualMap, new Autotiler.Behaviour {
					EdgesExtend = false,
					EdgesIgnoreOutOfLevel = false,
					PaddingIgnoreOutOfLevel = false
				}).TileGrid;
				tiles.Position = new Vector2((float)GroupBoundsMin.X - base.X, (float)GroupBoundsMin.Y - base.Y);
				Add(tiles);
			}
			TryToInitPosition();
		}

		private void TryToInitPosition() {
			if (MasterOfGroup) {
				foreach (PushPuzzleBlock item in Group) {
					if (!item.awake) {
						return;
					}
				}

				//figure out what moveToTarget does
				MoveToTarget();
			} else {
				master.TryToInitPosition();
			}
		}

		//only runs for master
		private void AddToGroupAndFindChildren(PushPuzzleBlock from) {
			if (from.X < (float)GroupBoundsMin.X) {
				GroupBoundsMin.X = (int)from.X;
			}
			if (from.Y < (float)GroupBoundsMin.Y) {
				GroupBoundsMin.Y = (int)from.Y;
			}
			if (from.Right > (float)GroupBoundsMax.X) {
				GroupBoundsMax.X = (int)from.Right;
			}
			if (from.Bottom > (float)GroupBoundsMax.Y) {
				GroupBoundsMax.Y = (int)from.Bottom;
			}
			from.HasGroup = true;
			from.OnDashCollide = OnDash;
			from.MasterOffset = from.Position - Position;
			Group.Add(from);
			//Logger.Log("Scuffed Helper", $"Adding block at {from.Position}, pos:{Position}, {MasterOffset}");
			Moves.Add(new KeyValuePair<PushPuzzleBlock, Vector2>(from, from.MasterOffset));
			
			if (from != this) {
				from.master = this;
			}

			foreach (PushPuzzleBlock entity in Scene.Tracker.GetEntities<PushPuzzleBlock>()) {
				if (!entity.HasGroup && entity.Index == Index && 
					(Scene.CollideCheck(new Rectangle((int)from.X - 1, (int)from.Y, (int)from.Width + 2, (int)from.Height), entity) || 
					Scene.CollideCheck(new Rectangle((int)from.X, (int)from.Y - 1, (int)from.Width, (int)from.Height + 2), entity))
					){
					//not recursion, just looking for all other entities
					AddToGroupAndFindChildren(entity);
				}
			}
		}

		//fix bug where you dash while block is moving
		//bug where block does not move properly should definitely be patched
			//maybe incorrect ordering?
		//for the future, this ordering always goes from the top left, it should change based on direction
		private DashCollisionResults OnDash(Player player, Vector2 direction) {
			//set all blocks to their target pos, to alleviate bugs
			//worth mentioning that only the masterofgroup's 

			if (MasterOfGroup) {
				//set all blocks to their target pos, to alleviate bugs
				MoveToTarget(true);
				

				dashDirection = direction;
				TargetPos = TargetPos + direction * moveDistance;

				//order dictionary
				//cannot switch vector2 :(
				//right
                if(direction == Vector2.UnitX) {
					Moves = Moves.OrderBy(key => -key.Key.Right).ToList();
				}
				//down?
				if(direction == Vector2.UnitY) {
					Moves = Moves.OrderBy(key => -key.Key.Bottom).ToList();
				}
				//left
				if (direction == -Vector2.UnitX) {
					Moves = Moves.OrderBy(key => key.Key.Left).ToList();
				}
				//up
				if (direction == -Vector2.UnitY) {
					Moves = Moves.OrderBy(key => key.Key.Top).ToList();
				}

				/*
				if (direction.X == 0) {
					Moves = Moves.OrderBy(key => key.Value.Y * -direction.Y).ToList();
                } else {
					Moves = Moves.OrderBy(key => key.Value.X * -direction.X).ToList();
				}
				*/

				/*
				Logger.Log("Scuffed Helper", $"{dashDirection}");
				foreach (KeyValuePair<PushPuzzleBlock, Vector2> move in Moves) {
					Logger.Log("Scuffed Helper", $"{move.Value}");
				}
				*/

					//let's just try this
			}
			//Logger.Log("ScuffedHelper", $"{dashDirection}");
			return DashCollisionResults.NormalOverride;
		}

		public override void Update() {
            base.Update();
			if (MasterOfGroup) {
				MoveToTarget();
            }
        }

		//Calc.Approach?
		private PushPuzzleBlock collidedBlock;
		bool collided;
		private void MoveToTarget(bool allTheWay = false) {
			Vector2 movement = (TargetPos - Position) / 6;
			if (allTheWay) {
				movement = TargetPos - Position;
            }
			
			collided = false;

			//Logger.Log("Scuffed Helper", "Moving to target");

			//block, offset
			foreach (KeyValuePair<PushPuzzleBlock, Vector2> move in Moves) {
				PushPuzzleBlock key = move.Key;
				Vector2 offset = move.Value;

				collidedBlock = key;

				key.MoveHCollideSolids(movement.X, false, keyOnCollide);
				key.MoveVCollideSolids(movement.Y, false, keyOnCollide);

				//shouldn't need to do this, but it fixes problems
				//just don't move the blocks too fast
                if (collided) {
					break;
                }
			}
			//Line up blocks, can mess up movement
			//you may be wondering "what is this 0.25 doing here?"
			//same
			if(Position == TargetPos) {
				foreach (KeyValuePair<PushPuzzleBlock, Vector2> move in Moves) {
					move.Key.MoveTo(Position + move.Value);
				}
			}
			
			
			
		}

		//try calling base.update()
		private void keyOnCollide(Vector2 v1, Vector2 v2, Platform p) {
			//if self collision, dont
			if(p.GetType() == typeof(PushPuzzleBlock)) {
				if (collidedBlock.master == (p as PushPuzzleBlock).master) {
					return;
                }
			}
			

			collided = true;

			Vector2 masterPosition = collidedBlock.Position - collidedBlock.MasterOffset;
			//move master to masterPosition
			master.TargetPos = masterPosition;
			master.MoveTo(masterPosition);

			//set all other target positions
			foreach (KeyValuePair<PushPuzzleBlock, Vector2> move in Moves) {
				PushPuzzleBlock ppb = move.Key;
				ppb.TargetPos = masterPosition + move.Value;
				//ppb.Position = ppb.TargetPos;
				ppb.MoveTo(ppb.TargetPos);
			}

			//Logger.Log("Scuffed Helper", $"Collided block: {collidedBlock.Position}");
		}

        #region rendering
        private Color interpolate(Color c1, Color c2, float pos) {
			Vector4 c1V = c1.ToVector4();
			Vector4 c2V = c2.ToVector4();
			return new Color(
				Vector4.Lerp(c1V, c2V, pos)
				);
        }

		bool advancedRender = false;
        public override void Render() {
			TimeSpan timeSinceLoad = DateTime.Now.Subtract(ScuffedHelperModule.LastLoadTime);
			//was 3e3
			double adjustedms = (timeSinceLoad.TotalMilliseconds / 3e3);

			const float scale = 10;
			const float warpScale = 5;

			Color lightColour = new Color(70, 139, 218, 128);
			Color darkColour = new Color(36, 8, 78, 200);

			//correct position, so that there aren't gaps
			Vector2 oldPosition = Position;
			foreach (KeyValuePair<PushPuzzleBlock, Vector2> move in master.Moves) {
				if(move.Key == this) {
					Position = master.Position + move.Value;
                }
			}

			//maybe try improving speed of perlin with a lookup
			if (advancedRender) {
				for (int x = 0; x < Width; x++) {
					for (int y = 0; y < Height; y++) {
						float currX = x + Position.X;
						float currY = y + Position.Y;

						float absX = Math.Abs(currX);
						float absY = Math.Abs(currY);
						//render each pixel individually
						//attempt to implement https://www.iquilezles.org/www/articles/warp/warp.htm

						//this apparently has negative space issues, so whoops
						//just take abs

						//Logger.Log("ScuffedHelper", $"{ticks}");
						//multiply X and Y by index, so that different blocks are different
						float offsetXNoise = (float)Perlin.perlin(absX / warpScale, absY / warpScale, adjustedms);
						//add 10 for this to be essentially different noise
						float offsetYNoise = (float)Perlin.perlin(absX / warpScale, absY / warpScale, adjustedms + 10);
						//add index, once again to change noise
						float noise = (float)Perlin.perlin(absX / scale + offsetXNoise, absY / scale + offsetYNoise, adjustedms + Index * 10);

						Color color = interpolate(lightColour, darkColour, noise);

						Draw.Point(new Vector2(currX, currY), color);



					}
				}
            } else {
				/*
                if (MasterOfGroup) {
					Draw.Rect(Collider, lightColour);
                } else {
					Draw.Rect(Collider, darkColour);
				}
				*/
				Draw.Rect(Collider, BlockColour);
				
				/*
				float noise = 0;
				foreach(KeyValuePair<PushPuzzleBlock, Vector2> move in master.Moves) {
					noise++;
					if(move.Key == this) {
						break;
                    }
                }
				noise /= master.Moves.Count;
				Draw.Rect(Collider, interpolate(lightColour, darkColour, noise));
				*/

				//Draw.HollowRect(TargetPos, Width, Height, Color.Red);
				
				
			}

			Position = oldPosition;

			try {
				//inneficient, since this gets rendered for each pushblock in the group
				master.Components.Get<TileGrid>().Render();
			}catch(NullReferenceException e) {
				//Pro code tip: not this
            }
			
            base.Render();
        }
        #endregion
    }
}
