﻿using Microsoft.Xna.Framework;
using Monocle;
using MonoMod.Cil;
using MonoMod.RuntimeDetour;
using System.Reflection;

namespace Celeste.Mod.MoreNeonline;

/// <summary>
/// Change various text colors on the save file tickets, to make not make them blend in. Black on black is hard to read. 
/// </summary>
/// <author>Snip</author>
public static class FileSelectNameColor
{
    private const string Name_OuiFileSelectSlot_Render = "orig_Render";
    private static readonly MethodInfo m_OuiFileSelectSlot_Render = typeof(OuiFileSelectSlot).GetMethod(Name_OuiFileSelectSlot_Render, BindingFlags.Public | BindingFlags.Instance);
    private static ILHook Hook_OuiFileSelectSlot_Render;

    private const string LogID = $"{nameof(MoreNeonlineModule)}/{nameof(FileSelectNameColor)}";

    public static void Load()
    {
        Hook_OuiFileSelectSlot_Render = new ILHook(m_OuiFileSelectSlot_Render, OuiFileSelectSlot_Render);
    }

    public static void Unload()
    {
        Hook_OuiFileSelectSlot_Render?.Dispose();
    }

    private static void OuiFileSelectSlot_Render(ILContext il)
    {
        ILCursor cursor = new(il);

        const string colorTargetCIL
            = "[call valuetype Microsoft.Xna.Framework.Color Microsoft.Xna.Framework.Color::get_Black()]";

        // button outline

        if (!cursor.TryGotoNext(MoveType.After,
            static instr => instr.MatchLdloc(5),
            static instr => instr.MatchLdfld<OuiFileSelectSlot.Button>("Label")
        ))
        {
            Logger.Log(LogLevel.Error, LogID, "Could not find label border render CIL!");
            Logger.Log(LogLevel.Error, LogID, "[ldloc.s 5, ldfld string Celeste.OuiFileSelectSlot/Button::Label]");
            return;
        }
        Logger.Log(LogLevel.Debug, LogID, $"Found label border render CIL @ {cursor.Instrs[cursor.Index - 2]}");

        if (!cursor.TryGotoNext(MoveType.After,
            static instr => instr.MatchCall<Color>("get_Black")
        ))
        {
            Logger.Log(LogLevel.Error, LogID, "Could not find Color.Black when rendering label border!");
            Logger.Log(LogLevel.Error, LogID, colorTargetCIL);
            return;
        }
        Logger.Log(LogLevel.Debug, LogID, $"Found Color.Black when rendering label border @ {cursor.Instrs[cursor.Index - 1]}");
        Logger.Log(LogLevel.Debug, LogID, $"Injecting {nameof(OverrideLabelBorderColor)} delegate @ {cursor.Next}");

        cursor.EmitDelegate(OverrideLabelBorderColor);


        // file time color

        if (!cursor.TryGotoNext(MoveType.After,
            static instr => instr.MatchLdarg0(),
            static instr => instr.MatchLdfld<OuiFileSelectSlot>("Time")
        ))
        {
            Logger.Log(LogLevel.Error, LogID, "Could not find file time render CIL!");
            Logger.Log(LogLevel.Error, LogID, "[ldarg.0, ldfld string Celeste.OuiFileSelectSlot::Time]");
            return;
        }
        Logger.Log(LogLevel.Debug, LogID, $"Found file time render CIL @ {cursor.Instrs[cursor.Index - 2]}");

        if (!cursor.TryGotoNext(MoveType.After,
            static instr => instr.MatchCall<Color>("get_Black")
        ))
        {
            Logger.Log(LogLevel.Error, LogID, "Could not find Color.Black when rendering file time!");
            Logger.Log(LogLevel.Error, LogID, colorTargetCIL);
            return;
        }
        Logger.Log(LogLevel.Debug, LogID, $"Found Color.Black when rendering file time @ {cursor.Instrs[cursor.Index - 1]}");
        Logger.Log(LogLevel.Debug, LogID, $"Injecting {nameof(OverrideTimeColor)} delegate @ {cursor.Next}");

        cursor.EmitDelegate(OverrideTimeColor);

        // filename color

        if (!cursor.TryGotoNext(MoveType.After,
            static instr => instr.MatchLdloc(24),
            static instr => instr.MatchLdloc(25),
            static instr => instr.MatchLdcR4(0.5f),
            static instr => instr.MatchLdcR4(1f),
            static instr => instr.MatchNewobj<Vector2>()
        ))
        {
            Logger.Log(LogLevel.Error, LogID, "Could not find filename render CIL!");
            Logger.Log(LogLevel.Error, LogID, "[ldloc.s 24, ldloc.s 25, ldc.r4 0.5f, ldc.r4 1.0f, newobj instance void Microsoft.Xna.Framework.Vector2::.ctor(float32, float32)]");
            return;
        }
        Logger.Log(LogLevel.Debug, LogID, $"Found filename render CIL @ {cursor.Instrs[cursor.Index - 5]}");

        if (!cursor.TryGotoNext(MoveType.After,
            instr => instr.MatchCall<Color>("get_Black")
        ))
        {
            Logger.Log(LogLevel.Error, LogID, "Could not find Color.Black when rendering filename!");
            Logger.Log(LogLevel.Error, LogID, colorTargetCIL);
            return;
        }
        Logger.Log(LogLevel.Debug, LogID, $"Found Color.Black when rendering filename @ {cursor.Instrs[cursor.Index - 1]}");
        Logger.Log(LogLevel.Debug, LogID, $"Injecting {nameof(OverrideNameColor)} delegate @ {cursor.Next}");

        cursor.EmitDelegate(OverrideNameColor);

        // chapter name color

        if (!cursor.TryGotoNext(MoveType.After,
            static instr => instr.MatchLdarg0(),
            static instr => instr.MatchLdfld<OuiFileSelectSlot>("FurthestArea")
        ))
        {
            Logger.Log(LogLevel.Error, LogID, "Could not find chapter name render CIL!");
            Logger.Log(LogLevel.Error, LogID, "[ldarg.0, ldfld int32 Celeste.OuiFileSelectSlot::FurthestArea]");
            return;
        }
        Logger.Log(LogLevel.Debug, LogID, $"Found chapter name render CIL @ {cursor.Instrs[cursor.Index - 2]}");

        if (!cursor.TryGotoNext(MoveType.After,
            static instr => instr.MatchCall<Color>("get_Black")
        ))
        {
            Logger.Log(LogLevel.Error, LogID, "Could not find Color.Black when rendering chapter name!");
            Logger.Log(LogLevel.Error, LogID, colorTargetCIL);
            return;
        }
        Logger.Log(LogLevel.Debug, LogID, $"Found Color.Black when rendering chapter name @ {cursor.Instrs[cursor.Index - 1]}");
        Logger.Log(LogLevel.Debug, LogID, $"Injecting {nameof(OverrideChapterNameColor)} delegate @ {cursor.Next}");

        cursor.EmitDelegate(OverrideChapterNameColor);
    }

    private static Color OverrideTimeColor(Color @default)
        => Color.Goldenrod;

    private static Color OverrideNameColor(Color @default)
        => Player.NormalHairColor;

    private static Color OverrideChapterNameColor(Color @default)
        => Player.UsedHairColor;

    private static Color OverrideLabelBorderColor(Color @default)
        => MoreNeonlineModule.Settings.EnableLabelPhaseEffect
            ? Calc.HsvToColor((Engine.Scene.TimeActive / 4f) % 1f, 1f, 0.5f)
            : @default;
}
